﻿using CefSharp;
using Genesyslab.Desktop.Infrastructure;
using Genesyslab.Desktop.Infrastructure.Commands;
using Genesyslab.Desktop.Infrastructure.DependencyInjection;
using Genesyslab.Desktop.Infrastructure.ViewManager;
using Genesyslab.Desktop.Modules.JourneyTimeline.Model;
using Genesyslab.Desktop.Modules.JourneyTimeline.View;
using Genesyslab.Desktop.Modules.JourneyTimeline.ViewModel;
using Genesyslab.Desktop.Modules.Windows.Event;
using Genesyslab.Platform.Commons.Logging;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

/**
 * IWS Plugin: Journey Timeline
 * Embeds a Chromium Browser into Interaction Workspace when handling interactions which have a Customer ID specified.
 * Customer ID is retrieved from the User Attached Data using a configurable key (default: customer_id)
 * This plugin has two configuration options:
 *  For Interaction Workspace Application Object, under the interaction-workspace section:
 *      [journeytimeline.id_key] = "customer_id"
 *      [journeytimeline.url] = ""
 *      
 * Note that to display a Journey Timeline page, the agent must have access privileges to GMS.
 */
namespace Genesyslab.Desktop.Modules.JourneyTimeline
{
    public class JourneyTimelineModule : IModule
    {
        private readonly IObjectContainer container;
        private readonly IViewManager viewManager;
        private readonly ILogger logger;

        public JourneyTimelineModule(IObjectContainer container, 
                                    IViewManager viewManager)
        {
            this.container = container;
            this.viewManager = viewManager;

            logger = container.Resolve<ILogger>();
            logger = logger.CreateChildLogger("JourneyTimeline");
            logger.Debug("--> JourneyTimelineModule.Initialize()");
        }

        public void Initialize()
        {
            // Register the JourneyTimeline configuration handler as a singleton
            container.RegisterType<IJourneyTimelineConfig, JourneyTimelineConfig>(true);
            // Register the JourneyTimeline view for Dependency Injection
            container.RegisterType<IJourneyTimelineViewModel, JourneyTimelineViewModel>();
            container.RegisterType<IJourneyTimelineView, JourneyTimelineView>();

            // Put the MySample view in the region "InteractionWorksheetRegion"
            viewManager.ViewsByRegionName["InteractionWorksheetRegion"].Add(
                new ViewActivator()
                {
                    ViewType = typeof(IJourneyTimelineView),
                    ViewName = "JourneyTimeline",
                    ActivateView = true,
                    Condition = JourneyTimelineViewModel.JourneyTimelineViewModelCondition
                }
            );

            // Here we register the view (GUI) "ISideButtonView"
            container.RegisterType<ISideButtonView, SideButtonView>();

            // Put the SideButtonView view in the region "CaseViewSideButtonRegion" (The case toggle button in the interaction windows)
            viewManager.ViewsByRegionName["CaseViewSideButtonRegion"].Add(
                new ViewActivator()
                {
                    ViewType = typeof(ISideButtonView),
                    ViewName = "SideButtonView",
                    ActivateView = true,
                    DynamicOnly = true
                }
            );

            // Initialize embedded chromium browser
            Cef.Initialize();

            logger.Debug("<-- JourneyTimelineModule.Initialize()");
        }
    }
}
