﻿using CefSharp;
using CefSharp.Wpf;
using Genesyslab.Desktop.Infrastructure.Configuration;
using Genesyslab.Desktop.Infrastructure.DependencyInjection;
using Genesyslab.Desktop.Infrastructure.ViewManager;
using Genesyslab.Desktop.Modules.Core.Model.Agents;
using Genesyslab.Desktop.Modules.Core.Model.Interactions;
using Genesyslab.Desktop.Modules.JourneyTimeline.Model;
using Genesyslab.Desktop.Modules.JourneyTimeline.Utility;
using Genesyslab.Platform.Commons.Logging;
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Web;
using System.Windows.Input;

namespace Genesyslab.Desktop.Modules.JourneyTimeline.ViewModel
{
    public class JourneyTimelineViewModel : ObservableObject, IJourneyTimelineViewModel
    {
        public BoundAuth myAuthInfo;
        private readonly IJourneyTimelineConfig mConfig;

        public JourneyTimelineViewModel(IObjectContainer container, IJourneyTimelineConfig config)
        {
            IAgent agent = container.Resolve<IAgent>();
            mConfig = config;
            PropertyChanged += OnPropertyChanged;

            // Retrieve Agent login credentials (for use in JourneyTimeline HTTP Authentication)
            myAuthInfo = new BoundAuth(agent.UserName, agent.Password);
        }

        public static bool JourneyTimelineViewModelCondition(ref object context)
        {
            IDictionary<string, object> contextDictionary = context as IDictionary<string, object>;
            object caseView;
            contextDictionary.TryGetValue("CaseView", out caseView);
            object caseObject;
            contextDictionary.TryGetValue("Case", out caseObject);
            ICase @case = caseObject as ICase;
            if (@case != null)
            {
                if (@case.MainInteraction != null)
                {
                    IJourneyTimelineConfig config = ContainerAccessPoint.Container.Resolve<IJourneyTimelineConfig>();
                    IInteraction i = @case.MainInteraction;
                    string customerId = i.GetAttachedData(config.getIdKey()) as string;
                    // Don't show JourneyTimeline panel if customer ID is not found.
                    if (!string.IsNullOrEmpty(customerId))
                    {
                        IViewManager viewManager = ContainerAccessPoint.Container.Resolve<IViewManager>();
                        viewManager.InstantiateDynamicViewInRegion(caseView, "InteractionWorksheetRegion", "JourneyTimeline", "JourneyTimeline");
                        viewManager.InstantiateDynamicViewInRegion(caseView, "CaseViewSideButtonRegion", "SideButtonView", "SideButtonView");
                    }
                }
            }
            return false;
        }

        #region Fields

        private ICase @case;
        public ICase Case
        {
            get { return @case; }
            set { @case = value; RaisePropertyChangedEvent("Case"); }
        }

        private string address;
        public string Address
        {
            get { return address; }
            set { address = value; RaisePropertyChangedEvent("Address"); }
        }

        private IWpfWebBrowser webBrowser;
        public IWpfWebBrowser WebBrowser
        {
            get { return webBrowser; }
            set { webBrowser = value; RaisePropertyChangedEvent("WebBrowser"); }
        }

        #endregion Fields

        protected void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            switch (e.PropertyName)
            {
                case "Case":
                    if (Case != null)
                    {
                        ReadCaseData();
                    }
                    break;
                case "WebBrowser":
                    if (WebBrowser != null)
                    {
                        WebBrowser.LoadError += OnWebBrowserLoadError;
                        // Bind Agent login details to JS object for GMS HTTP Authentication
                        WebBrowser.RegisterJsObject("genesysAuthObj", myAuthInfo);
                    }
                    break;
            }
        }

        /**
         * Read interaction CaseData to retrieve customer id and display Journey Timeline
         */
        protected void ReadCaseData()
        {
            IInteraction i = Case.MainInteraction;
            if (i == null)
            {
                return;
            }

            string customerId = i.GetAttachedData(mConfig.getIdKey()) as string;

            StringBuilder timelineAddress = new StringBuilder();
            timelineAddress.Append(mConfig.getUrl());
            // Build query-string parameter and ensure values are properly encoded
            timelineAddress.AppendFormat("?{0}={1}", HttpUtility.UrlEncode("customer"), HttpUtility.UrlEncode(customerId));
            Address = timelineAddress.ToString();
        }

        /**
         * Display error page in the event of page load failure
         */
        private void OnWebBrowserLoadError(object sender, LoadErrorEventArgs args)
        {
            // TODO: Page load retry timer
            if (args.ErrorCode == CefErrorCode.Aborted)
            {
                // Don't display error where user performed abort action
                return;
            }
            // Show error message (in our use case, it would be when the customer ID does
            // not yield a working Journey Timeline page)
            var errorMessage = "<html><body><h2>Failed to load URL " + args.FailedUrl +
                  " with error " + args.ErrorText + " (" + args.ErrorCode +
                  ").</h2></body></html>";

            WebBrowser.LoadHtml(errorMessage, args.FailedUrl);
        }
    }
}
