/*
 * Decompiled with CFR 0.152.
 */
package org.opensaml.xml.security.x509;

import java.security.GeneralSecurityException;
import java.security.cert.CRL;
import java.security.cert.CertPathBuilder;
import java.security.cert.CertPathBuilderException;
import java.security.cert.CertSelector;
import java.security.cert.CertStore;
import java.security.cert.CertStoreException;
import java.security.cert.Certificate;
import java.security.cert.CollectionCertStoreParameters;
import java.security.cert.PKIXBuilderParameters;
import java.security.cert.PKIXCertPathBuilderResult;
import java.security.cert.TrustAnchor;
import java.security.cert.X509CRL;
import java.security.cert.X509CertSelector;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import org.opensaml.xml.security.SecurityException;
import org.opensaml.xml.security.x509.CertPathPKIXValidationOptions;
import org.opensaml.xml.security.x509.InternalX500DNHandler;
import org.opensaml.xml.security.x509.PKIXTrustEvaluator;
import org.opensaml.xml.security.x509.PKIXValidationInformation;
import org.opensaml.xml.security.x509.PKIXValidationOptions;
import org.opensaml.xml.security.x509.X500DNHandler;
import org.opensaml.xml.security.x509.X509Credential;
import org.opensaml.xml.security.x509.X509Util;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class CertPathPKIXTrustEvaluator
implements PKIXTrustEvaluator {
    private final Logger log = LoggerFactory.getLogger(CertPathPKIXTrustEvaluator.class);
    private X500DNHandler x500DNHandler;
    private PKIXValidationOptions options;

    public CertPathPKIXTrustEvaluator() {
        this.options = new PKIXValidationOptions();
        this.x500DNHandler = new InternalX500DNHandler();
    }

    public CertPathPKIXTrustEvaluator(PKIXValidationOptions newOptions) {
        if (newOptions == null) {
            throw new IllegalArgumentException("PKIXValidationOptions may not be null");
        }
        this.options = newOptions;
        this.x500DNHandler = new InternalX500DNHandler();
    }

    @Override
    public PKIXValidationOptions getPKIXValidationOptions() {
        return this.options;
    }

    public X500DNHandler getX500DNHandler() {
        return this.x500DNHandler;
    }

    public void setX500DNHandler(X500DNHandler handler) {
        if (handler == null) {
            throw new IllegalArgumentException("X500DNHandler may not be null");
        }
        this.x500DNHandler = handler;
    }

    @Override
    public boolean validate(PKIXValidationInformation validationInfo, X509Credential untrustedCredential) throws SecurityException {
        if (this.log.isDebugEnabled()) {
            this.log.debug("Attempting PKIX path validation on untrusted credential: {}", (Object)X509Util.getIdentifiersToken(untrustedCredential, this.x500DNHandler));
        }
        try {
            PKIXBuilderParameters params = this.getPKIXBuilderParameters(validationInfo, untrustedCredential);
            this.log.trace("Building certificate validation path");
            CertPathBuilder builder = CertPathBuilder.getInstance("PKIX");
            PKIXCertPathBuilderResult buildResult = (PKIXCertPathBuilderResult)builder.build(params);
            if (this.log.isDebugEnabled()) {
                this.logCertPathDebug(buildResult, untrustedCredential.getEntityCertificate());
                this.log.debug("PKIX validation succeeded for untrusted credential: {}", (Object)X509Util.getIdentifiersToken(untrustedCredential, this.x500DNHandler));
            }
            return true;
        }
        catch (CertPathBuilderException e) {
            if (this.log.isTraceEnabled()) {
                this.log.trace("PKIX path construction failed for untrusted credential: " + X509Util.getIdentifiersToken(untrustedCredential, this.x500DNHandler), (Throwable)e);
            } else {
                this.log.error("PKIX path construction failed for untrusted credential: " + X509Util.getIdentifiersToken(untrustedCredential, this.x500DNHandler) + ": " + e.getMessage());
            }
            return false;
        }
        catch (GeneralSecurityException e) {
            this.log.error("PKIX validation failure", (Throwable)e);
            throw new SecurityException("PKIX validation failure", e);
        }
    }

    protected PKIXBuilderParameters getPKIXBuilderParameters(PKIXValidationInformation validationInfo, X509Credential untrustedCredential) throws GeneralSecurityException {
        Set<TrustAnchor> trustAnchors = this.getTrustAnchors(validationInfo);
        if (trustAnchors == null || trustAnchors.isEmpty()) {
            throw new GeneralSecurityException("Unable to validate X509 certificate, no trust anchors found in the PKIX validation information");
        }
        X509CertSelector selector = new X509CertSelector();
        selector.setCertificate(untrustedCredential.getEntityCertificate());
        this.log.trace("Adding trust anchors to PKIX validator parameters");
        PKIXBuilderParameters params = new PKIXBuilderParameters(trustAnchors, (CertSelector)selector);
        Integer effectiveVerifyDepth = this.getEffectiveVerificationDepth(validationInfo);
        this.log.trace("Setting max verification depth to: {} ", (Object)effectiveVerifyDepth);
        params.setMaxPathLength(effectiveVerifyDepth);
        CertStore certStore = this.buildCertStore(validationInfo, untrustedCredential);
        params.addCertStore(certStore);
        boolean isForceRevocationEnabled = false;
        boolean forcedRevocation = false;
        if (this.options instanceof CertPathPKIXValidationOptions) {
            CertPathPKIXValidationOptions certpathOptions = (CertPathPKIXValidationOptions)this.options;
            isForceRevocationEnabled = certpathOptions.isForceRevocationEnabled();
            forcedRevocation = certpathOptions.isRevocationEnabled();
        }
        if (isForceRevocationEnabled) {
            this.log.trace("PKIXBuilderParameters#setRevocationEnabled is being forced to: {}", (Object)forcedRevocation);
            params.setRevocationEnabled(forcedRevocation);
        } else if (this.storeContainsCRLs(certStore)) {
            this.log.trace("At least one CRL was present in cert store, enabling revocation checking");
            params.setRevocationEnabled(true);
        } else {
            this.log.trace("No CRLs present in cert store, disabling revocation checking");
            params.setRevocationEnabled(false);
        }
        return params;
    }

    protected boolean storeContainsCRLs(CertStore certStore) {
        Collection<? extends CRL> crls = null;
        try {
            crls = certStore.getCRLs(null);
        }
        catch (CertStoreException e) {
            this.log.error("Error examining cert store for CRL's, treating as if no CRL's present", (Throwable)e);
            return false;
        }
        return crls != null && !crls.isEmpty();
    }

    protected Integer getEffectiveVerificationDepth(PKIXValidationInformation validationInfo) {
        Integer effectiveVerifyDepth = validationInfo.getVerificationDepth();
        if (effectiveVerifyDepth == null) {
            effectiveVerifyDepth = this.options.getDefaultVerificationDepth();
        }
        return effectiveVerifyDepth;
    }

    protected Set<TrustAnchor> getTrustAnchors(PKIXValidationInformation validationInfo) {
        Collection<X509Certificate> validationCertificates = validationInfo.getCertificates();
        this.log.trace("Constructing trust anchors for PKIX validation");
        HashSet<TrustAnchor> trustAnchors = new HashSet<TrustAnchor>();
        for (X509Certificate cert : validationCertificates) {
            trustAnchors.add(this.buildTrustAnchor(cert));
        }
        if (this.log.isTraceEnabled()) {
            for (TrustAnchor anchor : trustAnchors) {
                this.log.trace("TrustAnchor: {}", (Object)anchor.toString());
            }
        }
        return trustAnchors;
    }

    protected TrustAnchor buildTrustAnchor(X509Certificate cert) {
        return new TrustAnchor(cert, null);
    }

    protected CertStore buildCertStore(PKIXValidationInformation validationInfo, X509Credential untrustedCredential) throws GeneralSecurityException {
        this.log.trace("Creating cert store to use during path validation");
        this.log.trace("Adding entity certificate chain to cert store");
        ArrayList<Object> storeMaterial = new ArrayList<Object>(untrustedCredential.getEntityCertificateChain());
        if (this.log.isTraceEnabled()) {
            for (X509Certificate cert : untrustedCredential.getEntityCertificateChain()) {
                this.log.trace(String.format("Added X509Certificate from entity cert chain to cert store with subject name '%s' issued by '%s' with serial number '%s'", this.x500DNHandler.getName(cert.getSubjectX500Principal()), this.x500DNHandler.getName(cert.getIssuerX500Principal()), cert.getSerialNumber().toString()));
            }
        }
        Date now = new Date();
        if (validationInfo.getCRLs() != null && !validationInfo.getCRLs().isEmpty()) {
            this.log.trace("Processing CRL's from PKIX info set");
            this.addCRLsToStoreMaterial(storeMaterial, validationInfo.getCRLs(), now);
        }
        if (untrustedCredential.getCRLs() != null && !untrustedCredential.getCRLs().isEmpty() && this.options.isProcessCredentialCRLs()) {
            this.log.trace("Processing CRL's from untrusted credential");
            this.addCRLsToStoreMaterial(storeMaterial, untrustedCredential.getCRLs(), now);
        }
        return CertStore.getInstance("Collection", new CollectionCertStoreParameters(storeMaterial));
    }

    protected void addCRLsToStoreMaterial(List<Object> storeMaterial, Collection<X509CRL> crls, Date now) {
        for (X509CRL crl : crls) {
            boolean isEmpty = crl.getRevokedCertificates() == null || crl.getRevokedCertificates().isEmpty();
            boolean isExpired = crl.getNextUpdate().before(now);
            if (!isEmpty || this.options.isProcessEmptyCRLs()) {
                if (!isExpired || this.options.isProcessExpiredCRLs()) {
                    storeMaterial.add(crl);
                    if (this.log.isTraceEnabled()) {
                        this.log.trace("Added X509CRL to cert store from issuer {} dated {}", (Object)this.x500DNHandler.getName(crl.getIssuerX500Principal()), (Object)crl.getThisUpdate());
                        if (isEmpty) {
                            this.log.trace("X509CRL added to cert store from issuer {} dated {} was empty", (Object)this.x500DNHandler.getName(crl.getIssuerX500Principal()), (Object)crl.getThisUpdate());
                        }
                    }
                    if (!isExpired) continue;
                    this.log.warn("Using X509CRL from issuer {} with a nextUpdate in the past: {}", (Object)this.x500DNHandler.getName(crl.getIssuerX500Principal()), (Object)crl.getNextUpdate());
                    continue;
                }
                if (!this.log.isTraceEnabled()) continue;
                this.log.trace("Expired X509CRL not added to cert store, from issuer {} nextUpdate {}", (Object)this.x500DNHandler.getName(crl.getIssuerX500Principal()), (Object)crl.getNextUpdate());
                continue;
            }
            if (!this.log.isTraceEnabled()) continue;
            this.log.trace("Empty X509CRL not added to cert store, from issuer {} dated {}", (Object)this.x500DNHandler.getName(crl.getIssuerX500Principal()), (Object)crl.getThisUpdate());
        }
    }

    private void logCertPathDebug(PKIXCertPathBuilderResult buildResult, X509Certificate targetCert) {
        this.log.debug("Built valid PKIX cert path");
        this.log.debug("Target certificate: {}", (Object)this.x500DNHandler.getName(targetCert.getSubjectX500Principal()));
        for (Certificate certificate : buildResult.getCertPath().getCertificates()) {
            this.log.debug("CertPath certificate: {}", (Object)this.x500DNHandler.getName(((X509Certificate)certificate).getSubjectX500Principal()));
        }
        TrustAnchor ta = buildResult.getTrustAnchor();
        if (ta.getTrustedCert() != null) {
            this.log.debug("TrustAnchor: {}", (Object)this.x500DNHandler.getName(ta.getTrustedCert().getSubjectX500Principal()));
        } else if (ta.getCA() != null) {
            this.log.debug("TrustAnchor: {}", (Object)this.x500DNHandler.getName(ta.getCA()));
        } else {
            this.log.debug("TrustAnchor: {}", (Object)ta.getCAName());
        }
    }
}

