/*
  GMSAPIUtil.m
  GMS Sample

 Disclaimer: IMPORTANT:  This software is supplied to you by Genesys
 Telecommunications Laboratories Inc ("Genesys") in consideration of your agreement
 to the following terms, and your use, installation, modification or redistribution
 of this Genesys software constitutes acceptance of these terms.  If you do not
 agree with these terms, please do not use, install, modify or redistribute this
 Genesys software.
 
 In consideration of your agreement to abide by the following terms, and subject
 to these terms, Genesys grants you a personal, non-exclusive license, under
 Genesys's copyrights in this original Genesys software (the "Genesys Software"), to
 use, reproduce, modify and redistribute the Genesys Software, with or without
 modifications, in source and/or binary forms; provided that if you redistribute
 the Genesys Software in its entirety and without modifications, you must retain
 this notice and the following text and disclaimers in all such redistributions
 of the Genesys Software.
 
 Neither the name, trademarks, service marks or logos of Genesys Inc. may be used
 to endorse or promote products derived from the Genesys Software without specific
 prior written permission from Genesys.  Except as expressly stated in this notice,
 no other rights or licenses, express or implied, are granted by Genesys herein,
 including but not limited to any patent rights that may be infringed by your
 derivative works or by other works in which the Genesys Software may be
 incorporated.
 
 The Genesys Software is provided by Genesys on an "AS IS" basis.  GENESYS MAKES NO
 WARRANTIES, EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION THE IMPLIED
 WARRANTIES OF NON-INFRINGEMENT, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 PURPOSE, REGARDING THE GENESYS SOFTWARE OR ITS USE AND OPERATION ALONE OR IN
 COMBINATION WITH YOUR PRODUCTS.
 
 IN NO EVENT SHALL GENESYS BE LIABLE FOR ANY SPECIAL, INDIRECT, INCIDENTAL OR
 CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 ARISING IN ANY WAY OUT OF THE USE, REPRODUCTION, MODIFICATION AND/OR
 DISTRIBUTION OF THE GENESYS SOFTWARE, HOWEVER CAUSED AND WHETHER UNDER THEORY OF
 CONTRACT, TORT (INCLUDING NEGLIGENCE), STRICT LIABILITY OR OTHERWISE, EVEN IF
 GENESYS HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 
 Copyright (C) 2013 Genesys Inc. All Rights Reserved.
 */

#import <Foundation/Foundation.h>
#import <UIKit/UIKit.h>
#import "GMSUtil.h"

static NSDateFormatter *dateFormatter;
static NSDateFormatter *localDateFormatter;

static NSString *baseURL;

@implementation GMSUtil

+ (NSString*) addParamsFromDictionary:(NSDictionary*)dict toString:(NSString*)base;
{
    __block NSString *paramStr = base;
    __block BOOL first = YES;
    [dict enumerateKeysAndObjectsUsingBlock:^(id key, id value, BOOL *stop){
        if (first)
            paramStr = [paramStr stringByAppendingFormat:@"%@=%@", key, value];
        else
            paramStr = [paramStr stringByAppendingFormat:@"&%@=%@", key, value];
        first = NO;
    }];
    return paramStr;
}

+(void) initialize
{
    dateFormatter = [[NSDateFormatter alloc] init];
    [dateFormatter setDateFormat:@"yyyy-MM-dd'T'HH:mm:ss.000'Z'"];
    [dateFormatter setTimeZone:[NSTimeZone timeZoneWithAbbreviation:@"GMT"]];
    
    localDateFormatter = [[NSDateFormatter alloc] init];
    [localDateFormatter setDateFormat:@"EEE LLL dd hh:mm a"];
    [localDateFormatter setTimeZone:[NSTimeZone localTimeZone]];
}

+(void) setBaseURL:(NSString*)URL
{
    baseURL = URL;
}

+ (NSDate*) dateFromApiString:(NSString*)dateString
{
    return [dateFormatter dateFromString:dateString];
}

+ (NSString*) apiStringFromDate:(NSDate*)date
{
    return [dateFormatter stringFromDate:date];
}

+ (NSString*) localStringFromDate:(NSDate*)date
{
    return [localDateFormatter stringFromDate:date];
}

+ (NSString*) urlParamsFromDictionary:(NSDictionary*)dict
{
    return [self addParamsFromDictionary:dict toString:@"?"];
}

+ (NSString*) bodyParamsFromDictionary:(NSDictionary*)dict
{
    return [self addParamsFromDictionary:dict toString:@""];
}

+ (NSString*) stringFromNumericValue:(NSObject*)value
{
    if ([value respondsToSelector:@selector(stringValue)])
        return [value performSelector:@selector(stringValue)];
    
    return @"";
}

+ (void) logRequest:(NSString *)txt2Log direction:(int)dir {
    //Update Log View
    [[NSNotificationCenter defaultCenter] postNotificationName:kUpdateNotification
                                                        object:self
                                                      userInfo:@{@"direction":@(dir), @"text":txt2Log}];
}

+ (void) submitRequestWithPath:(NSString *)path
                        method:(NSString *)method
                        params:(NSDictionary *)params
                       headers:(NSDictionary *)headers
               completionBlock:(void(^)(NSDictionary *responseDict))completionBlock {
    
    NSString *URL = [baseURL stringByAppendingString:path];
    [self submitRequestWithURL:URL method:method params:params headers:headers completionBlock:completionBlock];
}

+ (void) submitRequestWithPath:(NSString *)path
                        method:(NSString *)method
                        params:(NSDictionary *)params
                       headers:(NSDictionary *)headers
               responseBlock:(void(^)(NSDictionary *responseDict, NSError *error))responseBlock {
    
    NSString *URL = [baseURL stringByAppendingString:path];
    [self submitRequestWithURL:URL method:method params:params headers:headers responseBlock:responseBlock];
}

+ (void) submitRequestWithURL:(NSString *)URL
                       method:(NSString *)method
                       params:(NSDictionary *)params
                      headers:(NSDictionary *)headers
              completionBlock:(void(^)(NSDictionary *responseDict))completionBlock
{
    // Callers of this function are not expecting the completionBlock to be call if an error occured.
    // So we call submitRequestWithURL:method:params:headers:responseBlock and only callback if no error
    if (completionBlock != nil) {
        void(^block)(NSDictionary *responseDict, NSError *error) = ^(NSDictionary *responseDict, NSError *error){
            if (error == nil) {
                completionBlock(responseDict);
            }
        };
        [self submitRequestWithURL:URL method:method params:params headers:headers responseBlock:block];
    }
    else {
        [self submitRequestWithURL:URL method:method params:params headers:headers responseBlock:nil];
    }
}

+ (void) submitRequestWithURL:(NSString *)URL
                    method:(NSString *)method
                    params:(NSDictionary *)params
                   headers:(NSDictionary *)headers
           responseBlock:(void(^)(NSDictionary *responseDict, NSError *error))responseBlock
{
    if (URL == nil || method == nil) {
        //both path and method must be provided => return
        return;
    }
    
    [[UIApplication sharedApplication] setNetworkActivityIndicatorVisible:YES];
    
    NSMutableURLRequest *request;
    
    if ([method isEqualToString:@"GET"]) {
        
        NSString *urlWithParams = [URL stringByAppendingString:[GMSUtil urlParamsFromDictionary:params]];
        
        request = [NSMutableURLRequest requestWithURL:[NSURL URLWithString:urlWithParams]];
        [request setHTTPMethod:@"GET"];
        [request setValue:@"application/json" forHTTPHeaderField:@"Accept"];
    }
    else if ([method isEqualToString:@"POSTJSON"]) {
        NSError *error = nil;
        NSData *body = [NSJSONSerialization dataWithJSONObject:params options:kNilOptions error:&error];
        if(error)
            return; // Should be impossible. Means there is a coding error.
        
        request = [NSMutableURLRequest requestWithURL:[NSURL URLWithString:URL]];
        [request setHTTPMethod:@"POST"];
        [request setValue:@"application/json;charset=UTF-8" forHTTPHeaderField:@"Content-Type"];
        [request setHTTPBody:body];
        [request setValue:@"application/json" forHTTPHeaderField:@"Accept"];
    }
    else if ([method isEqualToString:@"POST"]) {
        request = [NSMutableURLRequest requestWithURL:[NSURL URLWithString:URL]];
        [request setHTTPMethod:@"POST"];
        [request setValue:@"application/json" forHTTPHeaderField:@"Accept"];
        
        //look if we can use application/x-www-form-urlencoded
        __block BOOL urlEncoding = YES;
        [params enumerateKeysAndObjectsUsingBlock:^(id key, id value, BOOL *stop){
            if ([value isKindOfClass:[NSData class]]) {
                urlEncoding = NO;
                *stop = YES;
            };
        }];
        
        if (urlEncoding) {
            
            [request setValue:@"application/x-www-form-urlencoded" forHTTPHeaderField:@"Content-Type"];
            
            NSString *paramStr = [GMSUtil bodyParamsFromDictionary:params];
            [request setHTTPBody:[paramStr dataUsingEncoding:NSUTF8StringEncoding]];
        }
        else { //Multi-part
            [request setValue:@"multipart/form-data; boundary=AaB03x" forHTTPHeaderField:@"Content-Type"];
            
            NSMutableData *requestBody = [[NSMutableData alloc] init];
            
            NSString *boundary = @"--AaB03x\r\nContent-Disposition: form-data; name=";
            NSDataBase64EncodingOptions base64Options = NSDataBase64Encoding76CharacterLineLength | NSDataBase64EncodingEndLineWithCarriageReturn | NSDataBase64EncodingEndLineWithLineFeed;
            
            [params enumerateKeysAndObjectsUsingBlock:^(id key, id value, BOOL *stop){
                if ([value isKindOfClass:[NSString class]]) {
                    NSString *part = [NSString stringWithFormat:@"%@\"%@\"\r\n\r\n%@\r\n", boundary, key, value];
                    [requestBody appendData:[part dataUsingEncoding:NSUTF8StringEncoding]];
                }
                if ([value isKindOfClass:[NSData class]]) {
                    NSString *part = [NSString stringWithFormat:@"%@\"%@\"; filename=\"GMS.png\"\r\nContent-Type:image/png\r\n", boundary, key];
                    [requestBody appendData:[part dataUsingEncoding:NSUTF8StringEncoding]];
                    [requestBody appendData:[@"Content-Transfer-Encoding: base64\r\n\r\n" dataUsingEncoding:NSUTF8StringEncoding]];
                    //NSData *debug = [((NSData *)value) base64EncodedDataWithOptions:base64Options];
                    //[requestBody appendData:[((NSData *)value) base64EncodedDataWithOptions:base64Options]];
                    NSData *debug = (NSData *)value;
                    [requestBody appendData:(NSData *)value];
                    [requestBody appendData:[@"\r\n" dataUsingEncoding:NSUTF8StringEncoding]];
                }
            }];
            
            [requestBody appendData:[@"--AaB03x--" dataUsingEncoding:NSUTF8StringEncoding]];
            [request setHTTPBody:requestBody];
        }
    }
    else if ([method isEqualToString:@"DELETE"]) {
        request = [NSMutableURLRequest requestWithURL:[NSURL URLWithString:URL]];
        [request setHTTPMethod:@"DELETE"];
    }
    
    request.cachePolicy = NSURLRequestUseProtocolCachePolicy;
    request.timeoutInterval = 30.0;
    [headers enumerateKeysAndObjectsUsingBlock:^(id key, id value, BOOL *stop){
        [request setValue:value forHTTPHeaderField:key];
    }];
    
    // Log the request
    NSString *strBody = [[NSString alloc] initWithData:request.HTTPBody encoding:NSASCIIStringEncoding];
    NSString *trimmedBody;
    trimmedBody = ([strBody length] > 350) ? [NSString stringWithFormat:@"%@ ... %@", [strBody substringToIndex:150], [strBody substringFromIndex:[strBody length] - 150]] : strBody;
    [self logRequest:[NSString stringWithFormat:@"Method: %@, URL: %@,\nHeaders: %@\nBody: %@\n", request.HTTPMethod, request.URL, [request allHTTPHeaderFields], trimmedBody] direction:toGMS];
    
    [NSURLConnection sendAsynchronousRequest:request
                                       queue:[NSOperationQueue mainQueue]
                           completionHandler:^(NSURLResponse *response, NSData *data, NSError *error) {
                               
               [[UIApplication sharedApplication] setNetworkActivityIndicatorVisible:NO];
               
               NSDictionary *responseDict = nil;
               
               if ([data length] > 0 && error == nil) {
                   NSString *responseStr = [[NSString alloc] initWithData:data encoding:NSUTF8StringEncoding];
                   [self logRequest:responseStr direction:fromGMS];
                   
                   // Try to parse JSON from the response. Not all responses contain JSON
                   NSError *jsonParsingError = nil;
                   responseDict = [NSJSONSerialization JSONObjectWithData:data options:0 error:&jsonParsingError];
               }
               else if (error != nil){
                   
                   UIAlertView *alert = [[UIAlertView alloc]
                                         initWithTitle: @"Error accessing GMS Server"
                                         message: [NSString stringWithFormat:@"URL: %@, Error: %@", URL, error]
                                         delegate: nil
                                         cancelButtonTitle:@"OK"
                                         otherButtonTitles:nil];
                   [alert show];
               }
               
               if (responseBlock) {
                   responseBlock(responseDict, error);
               }
           }
     ];
}



@end
