/*
 * Copyright (C) 2015 Genesys
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.genesys.gms.mobile.callback.demo.legacy.ui;

import android.content.SharedPreferences;
import android.os.Bundle;
import android.util.Log;

import com.genesys.gms.mobile.callback.demo.legacy.client.CometClient;
import com.genesys.gms.mobile.callback.demo.legacy.client.CometHandler;
import com.genesys.gms.mobile.callback.demo.legacy.data.api.pojo.ChatCometResponse;
import com.genesys.gms.mobile.callback.demo.legacy.data.api.pojo.ChatResponse;
import com.genesys.gms.mobile.callback.demo.legacy.data.api.pojo.ChatState;
import com.genesys.gms.mobile.callback.demo.legacy.data.api.pojo.TranscriptEntry;
import com.genesys.gms.mobile.callback.demo.legacy.data.events.chat.ChatDisconnectEvent;
import com.genesys.gms.mobile.callback.demo.legacy.data.events.chat.ChatRefreshEvent;
import com.genesys.gms.mobile.callback.demo.legacy.data.events.chat.ChatResponseEvent;
import com.genesys.gms.mobile.callback.demo.legacy.data.events.chat.ChatResponseEvent.ChatRequestType;
import com.genesys.gms.mobile.callback.demo.legacy.data.events.chat.ChatSendEvent;
import com.genesys.gms.mobile.callback.demo.legacy.data.events.chat.ChatStartEvent;
import com.genesys.gms.mobile.callback.demo.legacy.data.events.chat.ChatStartTypingEvent;
import com.genesys.gms.mobile.callback.demo.legacy.data.events.chat.ChatStopTypingEvent;
import com.genesys.gms.mobile.callback.demo.legacy.data.events.chat.ChatTranscriptEvent;
import com.genesys.gms.mobile.callback.demo.legacy.util.Globals;
import com.google.gson.Gson;
import com.google.gson.JsonSyntaxException;

import org.cometd.bayeux.Message;
import org.cometd.bayeux.client.ClientSessionChannel;
import org.eclipse.jetty.client.HttpClient;

import java.util.List;

import javax.inject.Inject;

import de.greenrobot.event.EventBus;
import hugo.weaving.DebugLog;
import timber.log.Timber;

public class GenesysChatController implements CometHandler {
  public static final String PROPERTY_REG_ID = "registration_id";
  private final SharedPreferences sharedPreferences;
  private final EventBus bus;
  private final Gson gson;

  private CometClient cometClient;

  // Should store below in a separate Model
  // TODO: Does this controller retain state following orientation change?
  private String sessionId;
  private String serverUrl;
  private String subject;

  @Override
  public String toString() {
    // For debugging
    return getClass().getName() + "@" + hashCode() +
        "[" +
        "cometClient=" + cometClient +
        ",sessionId=" + sessionId +
        ",serverUrl=" + serverUrl +
        ",subject=" + subject +
        "]";
  }

  /**
   * As the name implies, handles chat sessions. Communication with
   * chat server is conducted over Comet using the CometClient.
   * Note that dependency injection is facilitated by Dagger.
   * @param sharedPreferences
   * @param httpClient
   * @param gson
   */
  @Inject
  public GenesysChatController(SharedPreferences sharedPreferences,
                               HttpClient httpClient,
                               Gson gson) {
    this.sharedPreferences = sharedPreferences;
    this.cometClient = new CometClient(httpClient, this);
    this.gson = gson;
    this.bus = EventBus.getDefault();
  }

  public void persistState(Bundle outState) {
    // Nothing to do
  }

  public void restoreState(Bundle inState) {
    // From GenesysChatActivity
    sessionId = inState.getString("sessionId");
    serverUrl = inState.getString("cometUrl");
    if (serverUrl.startsWith("/")){
      Boolean isSecure = sharedPreferences.getBoolean(Globals.PROPERTY_PROTOCOL, Boolean.FALSE);
      String protocol = isSecure? "https":"http";
      serverUrl=protocol+"://"+sharedPreferences.getString(Globals.PROPERTY_HOST, "localhost")+":"+sharedPreferences.getString(Globals.PROPERTY_PORT,"8080")+serverUrl;
    }
    subject = inState.getString("subject");
  }

  // TODO: Functionality first, refactor later.

  // Must be Async
  public void startComet(String serverUrl) {
    this.serverUrl = serverUrl;
    String gmsUser = sharedPreferences.getString(Globals.PROPERTY_GMS_USER, null);
    String apikey = sharedPreferences.getString(Globals.PROPERTY_APIGEEKEY, null);
    if (serverUrl.startsWith("/")){
      Boolean isSecure = sharedPreferences.getBoolean(Globals.PROPERTY_PROTOCOL, Boolean.FALSE);
      String protocol = isSecure? "https":"http";
      serverUrl=protocol+"://"+sharedPreferences.getString(Globals.PROPERTY_HOST, "localhost")+":"+sharedPreferences.getString(Globals.PROPERTY_PORT,"8080")+serverUrl;
    }
    cometClient.start(serverUrl, gmsUser,apikey);
  }

  // Must be Async
  public void stopComet() {
    sharedPreferences.edit().remove("newMessages").apply();
    cometClient.disconnect();
  }

  /**
   * Issue a request to start a chat session. The request will be
   * serviced by the ChatApiManager, which issues an HTTP request
   * to GMS.
   * @param sessionId
   * @param subject
   */
  public void startChat(String sessionId, String subject) {
    this.sessionId = sessionId;
    this.subject = subject;
    Timber.d("Starting chat service.");
    bus.post(new ChatStartEvent(
        sessionId,
        true,
        "comet",
        sharedPreferences.getString("first_name", null),
        sharedPreferences.getString("last_name", null),
        sharedPreferences.getString("chat_email", null),
        subject,
        null, // subscriptionID
        sharedPreferences.getString("chat_display_name", null),
        sharedPreferences.getString(PROPERTY_REG_ID, null),
        "fcm", // pushNotificationType
        null, // pushNotificationLanguage
        false // pushNotificationDebug
    ));
  }

  public void disconnectChat() {
    if (sessionId != null && !sessionId.isEmpty()) {
      Timber.d("Disconnecting chat service.");
      bus.post(new ChatDisconnectEvent(sessionId, true));
    }
  }

  public void sendText(String text) {
    bus.post(new ChatSendEvent(sessionId, text, true));
  }

  public void startTyping() {
    bus.post(new ChatStartTypingEvent(sessionId, true));
  }

  public void stopTyping() {
    bus.post(new ChatStopTypingEvent(sessionId, true));
  }

  @Override
  public void onConnect() {
    // Bayeux client connected
    Timber.d("Comet client is connected.");
    if(sessionId != null && !sessionId.isEmpty()) {
      // If session was recovered, issue Refresh to retrieve missed transcripts
      bus.post(new ChatRefreshEvent(sessionId, cometClient.getTranscriptPosition(), null, true));
    }
  }

  @Override
  public void onDisconnect() {
    // Bayeux client disconnected
    Timber.d("Comet client is disconnected.");
  }

  /**
   * Invoked by CometClient when message is received over Comet on
   * subscribed channels. Message is received as JSON which is then
   * parsed into Chat transcripts if possible.
   * @param channel
   * @param message
   */
  @Override
  @DebugLog
  public void onMessage(ClientSessionChannel channel, Message message) {
    ChatCometResponse chatCometResponse = null;
    Timber.d("Message from CometD: %s", message.getJSON());
    try {
      chatCometResponse = gson.fromJson(message.getJSON(), ChatCometResponse.class);
      Timber.d("Parsed ChatCometResponse: %s", chatCometResponse);
    } catch (JsonSyntaxException e) {

      Timber.e(e, "Failed to parse CometD message.");
      return;
    }
    ChatResponse chatResponse = null;
    chatResponse = chatCometResponse.getData().getMessage();
    Integer transcriptPosition = chatResponse.getTranscriptPosition();
    if (transcriptPosition < cometClient.getTranscriptPosition()) {
      Log.d("GenesysChatController", "Comet client is ahead of server!");
    } else {
      cometClient.setTranscriptPosition(transcriptPosition);
    }
    List<TranscriptEntry> transcriptEntryList = chatResponse.getTranscriptToShow();
    if (transcriptEntryList != null) {
      for (TranscriptEntry entry : transcriptEntryList) {
        bus.post(new ChatTranscriptEvent(entry));
      }
    } else {
      Timber.d("No transcript to show!");
    }
    if (chatResponse.getChatIxnState() == ChatState.DISCONNECTED) {
      bus.post(new ChatResponseEvent(chatResponse, ChatRequestType.DISCONNECT));
    }
  }

  public void handleStart(ChatResponse chatResponse) {

  }

  public void handleSend(ChatResponse chatResponse) {

  }

  public void handleRefresh(ChatResponse chatResponse) {

  }

  public void handleStartTyping(ChatResponse chatResponse) {

  }

  public void handleStopTyping(ChatResponse chatResponse) {

  }

  public void handleDisconnect(ChatResponse chatResponse) {

  }
}
