/*
 * Copyright (C) 2015 Genesys
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.genesys.gms.mobile.callback.demo.legacy.client;

import com.genesys.gms.mobile.callback.demo.legacy.data.events.chatv2.*;
import com.genesys.gms.mobile.callback.demo.legacy.util.Globals;
import com.google.firebase.iid.FirebaseInstanceId;

import org.cometd.bayeux.Channel;
import org.cometd.bayeux.Message;
import org.cometd.bayeux.client.ClientSession;
import org.cometd.bayeux.client.ClientSessionChannel;
import org.cometd.client.BayeuxClient;
import org.cometd.client.transport.ClientTransport;
import org.cometd.client.transport.LongPollingTransport;
import org.eclipse.jetty.client.HttpClient;
import org.eclipse.jetty.client.api.Request;

import java.util.HashMap;
import java.util.Map;

import timber.log.Timber;

/**
 * Created by stau on 2/9/2015.
 */
public class CometClient {

  public static final String GENESYS_CHANNEL = "/_genesys";
  private static CometClient instance;
  private final HttpClient httpClient;
  private final ClientTransport clientTransport;
  private BayeuxClient bayeuxClient;
  private CometHandler handler;
  private int transcriptPosition = 0;
  private String mChannel;


  private static final String OP_REQUEST_CHAT    = "requestChat";
  private static final String OP_REQUEST_NOTIF   = "requestNotifications";
  private static final String OP_REFRESH         = "refresh";
  private static final String OP_SEND_MESSAGE    = "sendMessage";
  private static final String OP_SEND_MESSAGE1   = "send";
  private static final String OP_START_TYPING    = "startTyping";
  private static final String OP_STOP_TYPING     = "stopTyping";
  private static final String OP_DISCONNECT      = "disconnect";
  private static final String OP_PUSHURL         = "pushUrl";
  private static final String OP_CUSTOMNOTICE    = "customNotice";
  private static final String OP_UPDATEUSERDATA  = "updateData";
  private static final String OP_UPDATENICKNAME  = "updateNickname";

  private static final String PRM_ALIAS        = "alias";
  private static final String PRM_CHATID       = "chatId";
  private static final String PRM_SECURE_KEY   = "secureKey";
  private static final String PRM_USER_ID      = "userId";

  private static final String PRM_MESSAGE      = "message";
  private static final String PRM_MESSAGE_TYPE = "messageType";
  private static final String PRM_PUSHURL      = "pushUrl";
  private static final String PRM_NICKNAME     = "nickname";
  private static final String PRM_TRS_POS      = "transcriptPosition";

  private static final String PRM_OPERATION   = "operation";
  private static final String PRM_FIRSTNAME   = "firstName";
  private static final String PRM_LASTNAME    = "lastName";
  private static final String PRM_SUBJECT     = "subject";
  private static final String PRM_USERDATA    = "userData";

  private boolean connected = false;

  public static void createCometClient(HttpClient httpClient,
                                        ClientTransport clientTransport,
                                        CometHandler handler) {
      instance = new CometClient(httpClient, clientTransport, handler);
  }

  public static CometClient getInstance() {
      return instance;
  }

  private CometClient(HttpClient httpClient, ClientTransport clientTransport, CometHandler handler) {
      this.httpClient = httpClient;
      this.clientTransport = clientTransport;
      this.handler = handler;
      this.bayeuxClient = null;
  }

  // TODO: This kind of logic should be removed from a generic Comet client wrapper
  public int getTranscriptPosition() {
    return transcriptPosition;
  }

  // TODO: This kind of logic should be removed from a generic Comet client wrapper
  public void setTranscriptPosition(int transcriptPosition) {
    this.transcriptPosition = transcriptPosition;
  }

  public void start(String serverUrl,String channel, final String gmsUser, final String apikey) {
    if (bayeuxClient != null && bayeuxClient.isConnected()) {
      // TODO: Bayeux Client is unexpectedly already connected?
      return;
    }
    mChannel = channel;
    if (bayeuxClient == null) {
      Map<String, Object> options = new HashMap<String, Object>();
      ClientTransport transport = new LongPollingTransport(options, httpClient) {
        @Override
        protected void customize(Request request) {
          if (gmsUser != null && !gmsUser.isEmpty()) {
            Timber.d("Adding %s: %s header.", Globals.GMS_USER_HEADER, gmsUser);
            request.header(Globals.GMS_USER_HEADER, gmsUser);
          }
          if (apikey != null && !apikey.isEmpty()) {
            Timber.d("Adding %s: %s header.", Globals.APIGEEKEY, gmsUser);
              request.header(Globals.APIGEEKEY, apikey);
          }
        }
      };

      if (null != mChannel) {
          bayeuxClient = new BayeuxClient(serverUrl,clientTransport, transport);
          addV2listeners();
      } else {
          bayeuxClient = new BayeuxClient(serverUrl, transport);
          addListeners();
          addExtensions();
      }

    }

    if (null == mChannel) {
        if (!bayeuxClient.isHandshook()) {
            bayeuxClient.handshake();

            /*
            boolean handshakeSuccess = bayeuxClient.waitFor(15000, BayeuxClient.State.CONNECTED);
            if (!handshakeSuccess)
                throw new RuntimeException("CometD handshake did not succeed");

            bayeuxClient.getChannel(GENESYS_CHANNEL).subscribe(new ClientSessionChannel.MessageListener() {
                @Override public void onMessage(ClientSessionChannel channel, Message message) {
                    try {
                        handler.onMessage(channel, message);
                    } catch (Exception e) {
                        Timber.e(e, "Error handling comet message.");
                    }
                }
            });
            */
        }
    }
  }


  protected void addV2listeners() {
      bayeuxClient.batch(new Runnable() {
          @Override
          public void run() {
              bayeuxClient.handshake(new ClientSessionChannel.MessageListener() {

                  @Override
                  public void onMessage(ClientSessionChannel channel, Message message) {
                      if (message.isSuccessful()) {
                          Timber.d("Chat V2 Handshake successful");
                          addV2Subscriptions();
                      }
                  }
              });
          }
      });
  }

  protected void addListeners() {
    bayeuxClient.batch(new Runnable() {
      @Override
      public void run() {
        bayeuxClient.getChannel(Channel.META_HANDSHAKE).addListener(new ClientSessionChannel.MessageListener() {
          @Override
          public void onMessage(ClientSessionChannel channel, Message message) {
            if (message.isSuccessful()) {
              Timber.d("Handshake successful, adding subs.");
              addSubscriptions();
            } else {
              Timber.e("Handshake did not succeed.");
            }
          }
        });
        bayeuxClient.getChannel(Channel.META_CONNECT).addListener(new ClientSessionChannel.MessageListener() {
          @Override
          public void onMessage(ClientSessionChannel channel, Message message) {
            if (bayeuxClient.isDisconnected()) {
              return;
            }
            boolean wasConnected = connected;
            connected = message.isSuccessful();
            if (!wasConnected && connected) {
              handler.onConnect();
            } else if (wasConnected && !connected) {
              handler.onDisconnect();
            }
          }
        });
        bayeuxClient.getChannel(Channel.META_DISCONNECT).addListener(new ClientSessionChannel.MessageListener() {
          @Override
          public void onMessage(ClientSessionChannel channel, Message message) {
            if (message.isSuccessful()) {
              connected = false;
              // TODO: Does this occur before/after META_CONNECT message?
            }
          }
        });
      }
    });
  }

  protected void addV2Subscriptions() {
      bayeuxClient.batch(new Runnable() {
          @Override
          public void run() {
              bayeuxClient.getChannel(mChannel).subscribe(
                      new ClientSessionChannel.MessageListener() {

                          @Override
                          public void onMessage(ClientSessionChannel channel, Message message) {
                              handler.onNewMessage(message);
                          }
                      }
              );
          }
      });
  }

  protected void addSubscriptions() {
    // Perform channel subscriptions here
    // Call hook to perform subscriptions
    bayeuxClient.batch(new Runnable() {
      @Override
      public void run() {
        bayeuxClient.getChannel(GENESYS_CHANNEL).subscribe(new ClientSessionChannel.MessageListener() {
          @Override
          public void onMessage(ClientSessionChannel channel, Message message) {
            handler.onMessage(channel, message);
          }
        });
      }
    });
  }

  protected void addExtensions() {
    bayeuxClient.addExtension(new ClientSession.Extension.Adapter() {
      @Override
      public boolean sendMeta(ClientSession session, Message.Mutable message) {
        if (Channel.META_DISCONNECT.equals(message.getChannel()) || Channel.META_CONNECT.equals(message.getChannel())) {
          Timber.d("Inserting transcriptPosition ext=%d", transcriptPosition);
          Map<String, Object> ext = message.getExt(true);
          // TODO: Use hook to retrieve value
          ext.put("transcriptPosition", Integer.toString(transcriptPosition));
        }
        return true;
      }
    });
  }

  public void disconnect() {
    if (bayeuxClient != null && !bayeuxClient.isDisconnected()) {
      bayeuxClient.disconnect();
    }
  }

    public void requestChat(ChatV2StartEvent event){
        Map<String, Object> requestChatData = new HashMap<>();
        requestChatData.put(PRM_OPERATION, OP_REQUEST_CHAT);
        requestChatData.put(PRM_FIRSTNAME, event.firstName);
        requestChatData.put(PRM_LASTNAME, event.lastName);
        requestChatData.put(PRM_NICKNAME, event.firstName);
        requestChatData.put(PRM_SUBJECT, event.subject);
        requestChatData.put(PRM_USERDATA, getSubscriberData());
        publish(requestChatData);
    }

    public void startOpTyping(ChatV2StartTypingEvent event) {
        Map<String, Object> requestChatData = new HashMap<>();
        requestChatData.put(PRM_OPERATION, OP_START_TYPING);
        requestChatData.put(PRM_MESSAGE, event.s.toString());
        requestChatData.put(PRM_CHATID, event.chatId);
        requestChatData.put(PRM_USER_ID, event.userId);
        requestChatData.put(PRM_SECURE_KEY, event.secureKey);
        publish(requestChatData);
    }

    public void stopOpTyping(ChatV2StopTypingEvent event) {
        Map<String, Object> requestChatData = new HashMap<>();
        requestChatData.put(PRM_OPERATION, OP_STOP_TYPING);
        requestChatData.put(PRM_MESSAGE, "typing Ended");
        requestChatData.put(PRM_CHATID, event.chatId);
        requestChatData.put(PRM_USER_ID, event.userId);
        requestChatData.put(PRM_SECURE_KEY, event.secureKey);
        publish(requestChatData);
    }

    public void sendOpMessage(ChatV2SendEvent event) {
        Map<String, Object> requestChatData = new HashMap<>();
        requestChatData.put(PRM_OPERATION, OP_SEND_MESSAGE);
        requestChatData.put(PRM_MESSAGE, event.text);
        requestChatData.put(PRM_CHATID, event.chatId);
        requestChatData.put(PRM_USER_ID, event.userId);
        requestChatData.put(PRM_SECURE_KEY, event.secureKey);
        publish(requestChatData);
    }

    public void disconnectChatV2(ChatV2DisconnectEvent event){
        Map<String, Object> requestChatData = new HashMap<>();
        requestChatData.put(PRM_OPERATION, OP_DISCONNECT);
        requestChatData.put(PRM_ALIAS, event.alias);
        requestChatData.put(PRM_CHATID, event.chatId);
        requestChatData.put(PRM_USER_ID, event.userId);
        requestChatData.put(PRM_SECURE_KEY, event.secureKey);
        publish(requestChatData);
    }

/* TODO: get actual user data
    public void updateOpMessage(String secureKey, String chatId, String userId){
        Map<String, Object> requestChatData = new HashMap<>();
        requestChatData.put(PRM_OPERATION, OP_UPDATEUSERDATA);
        requestChatData.put(PRM_CHATID, chatId);
        requestChatData.put(PRM_USER_ID, userId);
        requestChatData.put(PRM_SECURE_KEY, secureKey);
        requestChatData.put(PRM_USERDATA,getUserdata());
        publish(requestChatData);
    }


    private Map<String, Object> getUserdata() {
        Map<String, Object> result = new HashMap<>();
        result.put("key1", "test");
        result.put("key2", "fcm");
        result.put("key3", "en_us");

        return result;
    }

*/
    private Map<String, Object> getSubscriberData() {
        String regToken = FirebaseInstanceId.getInstance().getToken();
        Map<String, Object> result = new HashMap<>();
        result.put("push_notification_deviceid", regToken);
        result.put("push_notification_type", "fcm");
        result.put("push_notification_language", "en_us");

        return result;
    }

    private void publish(final Object data){
        bayeuxClient.batch(new Runnable() {
            @Override
            public void run() {
                bayeuxClient.getChannel(mChannel).publish(data, new ClientSessionChannel.MessageListener() {
                    @Override
                    public void onMessage(ClientSessionChannel channel, Message message) {
                        handler.onNewMessage(message);
                    }
                });
            }
        });
    }
}
