/*
 * Copyright (C) 2015 Genesys
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.genesys.gms.mobile.callback.demo.legacy.ui;

import android.app.AlertDialog;
import android.app.Notification;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.SharedPreferences;
import android.graphics.Color;
import android.graphics.Typeface;
import android.net.Uri;
import android.os.AsyncTask;
import android.os.Bundle;
import android.os.Environment;
import android.support.annotation.NonNull;
import android.support.v4.app.NotificationCompat;
import android.support.v4.app.NotificationManagerCompat;
import android.text.Editable;
import android.text.Spannable;
import android.text.SpannableString;
import android.text.TextWatcher;
import android.text.method.LinkMovementMethod;
import android.text.method.ScrollingMovementMethod;
import android.text.style.ClickableSpan;
import android.text.style.ForegroundColorSpan;
import android.text.style.StyleSpan;
import android.view.KeyEvent;
import android.view.Menu;
import android.view.MenuItem;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.inputmethod.EditorInfo;
import android.widget.EditText;
import android.widget.TextView;
import android.widget.TextView.OnEditorActionListener;
import android.widget.Toast;

import com.genesys.gms.mobile.callback.demo.legacy.R;
import com.genesys.gms.mobile.callback.demo.legacy.common.BaseActivity;
import com.genesys.gms.mobile.callback.demo.legacy.data.api.pojo.ChatV2Response;
import com.genesys.gms.mobile.callback.demo.legacy.data.api.pojo.TranscriptEntry;
import com.genesys.gms.mobile.callback.demo.legacy.data.events.chat.ChatErrorEvent;
import com.genesys.gms.mobile.callback.demo.legacy.data.events.chat.ChatResponseEvent;
import com.genesys.gms.mobile.callback.demo.legacy.data.events.chat.ChatTranscriptEvent;
import com.genesys.gms.mobile.callback.demo.legacy.data.events.chatv2.ChatV2TranscriptEvent;
import com.genesys.gms.mobile.callback.demo.legacy.data.events.filemgmt.FileDownloadRequestEvent;
import com.genesys.gms.mobile.callback.demo.legacy.data.events.filemgmt.FileDownloadedEvent;
import com.genesys.gms.mobile.callback.demo.legacy.data.events.filemgmt.FileUploadResponseEvent;
import com.genesys.gms.mobile.callback.demo.legacy.util.Globals;
import com.genesys.gms.mobile.callback.demo.legacy.util.PathUtil;

import java.io.File;
import java.net.URISyntaxException;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

import javax.inject.Inject;

import de.greenrobot.event.EventBus;
import hugo.weaving.DebugLog;
import timber.log.Timber;

public class GenesysChatActivity extends BaseActivity {

  private static final ScheduledExecutorService timer = Executors.newScheduledThreadPool(2);
  private static final int NID_CHAT_ACTIVY = 10;
  private static final int SELECT_FILE = 10;

  private final Executor uiExecutor = new Executor() {
    @Override
    public void execute(@NonNull Runnable command) {
      runOnUiThread(command);
    }
  };

  @Inject
  SharedPreferences sharedPreferences;
  @Inject
  GenesysChatController controller;
  private final EventBus bus;

  private static final int FILE_SELECT_CODE = 0;

  private TextView transcriptTextView;
  private View sendButton;
  private View transferButton;
  private EditText sendEditText;
  private TextView infoTextView;
  private TextWatcher textWatcher;

  private boolean chatFinished;
  private boolean userTyping;

  private String cometUrl;
  private String sessionId;
  private String subject;

  @DebugLog
  public GenesysChatActivity() {
    this.bus = EventBus.getDefault();
  }

  /**
   * In this lifecycle callback, an attempt is made to restore a
   * previous chat session, if one existed. However, if no persisted
   * state is found, and if the intent specifies a START_CHAT action,
   * a request will be made to initiate the chat session.
   * @param inState
   */
  @Override
  @DebugLog
  protected void onCreate(Bundle inState) {
    super.onCreate(inState);

    final Intent intent = getIntent();

    if (inState == null) {
      if (Globals.ACTION_GENESYS_START_CHAT.equals(intent.getAction())) {
        cometUrl = intent.getStringExtra(Globals.EXTRA_COMET_URL);
        sessionId = intent.getStringExtra(Globals.EXTRA_SESSION_ID);
        subject = intent.getStringExtra(Globals.EXTRA_SUBJECT);
        if (sessionId != null && subject != null) {
          controller.startChat(sessionId, subject);
        }

        // Persist state in case of unexpected app termination
        sharedPreferences.edit()
            .putBoolean("CHAT_chatFinished", chatFinished)
            .putString("CHAT_cometUrl", cometUrl)
            .putString("CHAT_sessionId", sessionId)
            .putString("CHAT_subject", subject)
            .apply();
      }
    }

    setupUi(inState);

    if (inState == null) {
      Timber.d("Attempt to restore Chat state from persistence.");
      inState = new Bundle();
      inState.putBoolean("chatFinished", sharedPreferences.getBoolean("CHAT_chatFinished", false));
      inState.putString("cometUrl", sharedPreferences.getString("CHAT_cometUrl", null));
      inState.putString("sessionId", sharedPreferences.getString("CHAT_sessionId", null));
      inState.putString("subject", sharedPreferences.getString("CHAT_subject", null));
    }
    chatFinished = inState.getBoolean("chatFinished");
    cometUrl = inState.getString("cometUrl");
    sessionId = inState.getString("sessionId");
    subject = inState.getString("subject");

    controller.restoreState(inState);
    bus.register(this);
    if (cometUrl != null && !cometUrl.isEmpty()) {
        // TODO: Using RxJava could help us remove these...
        new AsyncTask<Void, Void, Void>() {
              @Override
              protected Void doInBackground(Void... params) {
                controller.startComet(cometUrl);
                return null;
            }
          }.execute();
    }
  }

  @Override
  public boolean onCreateOptionsMenu(Menu menu) {
    getMenuInflater().inflate(R.menu.chat_actions, menu);
    return true;
  }

  public void onEvent(FileDownloadedEvent event){

    Intent intent = new Intent();
    intent.setAction(android.content.Intent.ACTION_VIEW);
    File file = new File(event.getFilePath());
    intent.setDataAndType(Uri.fromFile(file), "*/*");

    PendingIntent pIntent = PendingIntent.getActivity(this, 0, intent, 0);

    Notification noti = new NotificationCompat.Builder(this)
            .setContentTitle("Download completed")
            .setContentText(event.getFilePath())
            .setSmallIcon(R.drawable.ic_launcher)
            .setContentIntent(pIntent).build();

    noti.flags |= Notification.FLAG_AUTO_CANCEL;

    NotificationManager notificationManager = (NotificationManager) getSystemService(NOTIFICATION_SERVICE);
    notificationManager.notify(0, noti);
    Toast.makeText(this,"Download completed",Toast.LENGTH_SHORT).show();
  }

  @Override
  protected void onSaveInstanceState(Bundle outState) {
    outState.putBoolean("chatFinished", chatFinished);
    outState.putString("cometUrl", cometUrl);
    outState.putString("sessionId", sessionId);
    outState.putString("subject", subject);
    outState.putCharSequence("transcript", transcriptTextView.getText());
    outState.putString("sendEditText", sendEditText.getText().toString());
    controller.persistState(outState);

    super.onSaveInstanceState(outState);
  }

  /**
   * Re-establish comet connection when application is restored from background
   */
  @Override
  protected void onResume() {
    super.onResume();

    NotificationManagerCompat.from(this).cancel(NID_CHAT_ACTIVY);
    if (chatFinished) {
      finishChat();
    }
  }

  @Override
  protected void onPause() {
    if (userTyping) {
      controller.stopTyping();
      if (scheduledStopTypingMessage != null && !scheduledStopTypingMessage.isDone()) {
        scheduledStopTypingMessage.cancel(false);
        scheduledStopTypingMessage = null;
      }
    }
      if (!chatFinished) {
          generateNotification();
      }

    super.onPause();
  }

  @Override
  @DebugLog
  protected void onDestroy() {
    new AsyncTask<Void, Void, Void>() {
        @Override
        protected Void doInBackground(Void... params) {
            controller.stopComet();
            return null;
        }
    }.execute();
    bus.unregister(this);

    NotificationManagerCompat.from(this).cancel(NID_CHAT_ACTIVY);

    super.onDestroy();
  }

    private void showFileChooser() {
    Intent chooser = new Intent(Intent.ACTION_GET_CONTENT);
    Uri uri = Uri.parse(Environment.getDownloadCacheDirectory().getPath().toString());
    chooser.addCategory(Intent.CATEGORY_OPENABLE);
    chooser.setDataAndType(uri, "*/*");
    // startActivity(chooser);
    try {
      startActivityForResult(chooser, SELECT_FILE);
    }
    catch (android.content.ActivityNotFoundException ex)
    {
      Toast.makeText(this, "Please install a File Manager.",
              Toast.LENGTH_SHORT).show();
    }
  }


  public void onActivityResult(int requestCode, int resultCode, Intent data)
  {
    if (resultCode == RESULT_OK)
    {
      Uri uri = data.getData();
      Toast.makeText(this,this.getContentResolver().getType(uri),Toast.LENGTH_LONG);
      //String path = getPath(this.getApplicationContext(), uri);
      //String path =   DocumentsContract.getDocumentId(uri);
      String path="";
      try {
        path = PathUtil.getPath(this.getApplicationContext(), uri);
        Toast.makeText(this,this.getContentResolver().getType(uri),Toast.LENGTH_LONG);

      } catch (URISyntaxException e) {
        e.printStackTrace();
      }
      transferFile(path);

    }
  }

  /**
   * Sets up listeners on chat elements, including a text watcher which
   * helps to produce the ${user} is typing message when user begins to
   * enter text in the text box.
   * @param inState
   */
  private void setupUi(Bundle inState) {
    setContentView(R.layout.chat_layout);

    sendEditText = (EditText) findViewById(R.id.sendText);
    transcriptTextView = (TextView) findViewById(R.id.transcriptText);
    sendButton = findViewById(R.id.sendButton);
    transferButton = findViewById(R.id.transferButton);
    //Used to disable Chat Transfert button in case of Chat V1
    String scenario = sharedPreferences.getString("scenario", null);

    if (scenario.equals("CHAT-V2")){
      transferButton.setEnabled(true);
    }else
      transferButton.setEnabled(false);

    infoTextView = (TextView) findViewById(R.id.informationalMessageTextView);

    if (inState != null) {
      transcriptTextView.setText(inState.getCharSequence("transcript"));
      sendEditText.setText(inState.getString("sendEditText"));
    }

    sendEditText.setOnEditorActionListener(new OnEditorActionListener() {
      @Override
      public boolean onEditorAction(TextView v, int actionId, KeyEvent event) {
        if (actionId == EditorInfo.IME_ACTION_SEND) {
          sendMessage();
          return true;
        }
        return false;
      }
    });

    textWatcher = new TextWatcher() {
      @Override
      @DebugLog
      public void onTextChanged(CharSequence s, int start, int before, int count) {
        updateSendButtonState();
      }

      @Override
      @DebugLog
      public void beforeTextChanged(CharSequence s, int start, int count, int after) {
        if (!userTyping && after > count) {
          userTyping = true;
        } else if (scheduledStopTypingMessage != null && !scheduledStopTypingMessage.isDone()) {
          scheduledStopTypingMessage.cancel(false);
          scheduledStopTypingMessage = null;
        }
      }

      @Override
      @DebugLog
      public void afterTextChanged(Editable s) {

        if(userTyping) controller.startTyping(s);

        if (scheduledStopTypingMessage == null || scheduledStopTypingMessage.isDone()) {
          scheduledStopTypingMessage = timer.schedule(updateUserTyping, 5, TimeUnit.SECONDS);
        }
      }
    };
    sendEditText.addTextChangedListener(textWatcher);

    transcriptTextView.setMovementMethod(new ScrollingMovementMethod());
    sendButton.setOnClickListener(new OnClickListener() {
      @Override
      public void onClick(View v) {
        sendMessage();
      }
    });
    infoTextView.setTextColor(Color.GRAY);
  }

  private void updateSendButtonState() {
    sendButton.setEnabled(!chatFinished && sendEditText.length() > 0);
  }

  private void transferFile(String file){
    controller.transferFile(file);
  }

  private void sendMessage() {
    stopTyping();
    final String text = sendEditText.getText().toString();
    sendEditText.setText("");
    controller.sendText(text);
  }

  private void stopTyping() {
    if (userTyping) {
      if (scheduledStopTypingMessage != null) {
        if (!scheduledStopTypingMessage.isDone()) {
          scheduledStopTypingMessage.cancel(false);
        }
        scheduledStopTypingMessage = null;
      }
      userTyping = false;
      controller.stopTyping();
    }
  }

  /**
   * Add new message to chat log
   * @param tag
   * @param message
   */
  private void appendTranscriptMessage(final String tag, final String message) {
    SpannableString text = new SpannableString(tag + ": " + message + "\n");
    text.setSpan(
        new ForegroundColorSpan(Color.GRAY),
        0, tag.length() + 1, Spannable.SPAN_EXCLUSIVE_EXCLUSIVE);
    transcriptTextView.append(text);
  }

  /**
   * Show info message below chat log
   * @param message
   */
  private void appendTranscriptInfo(final String message) {
    SpannableString text = new SpannableString(message);
    text.setSpan(new ForegroundColorSpan(Color.GRAY), 0, text.length(), Spannable.SPAN_EXCLUSIVE_EXCLUSIVE);
    text.setSpan(new StyleSpan(Typeface.ITALIC), 0, text.length(), Spannable.SPAN_EXCLUSIVE_EXCLUSIVE);
    transcriptTextView.append(text);
  }

    /**
     * Show link in the transcript to download later
     * @param event
     */
    private void appendDownloadLink(final FileDownloadRequestEvent event) {
        final SpannableString text = new SpannableString(" Download\n");

        ClickableSpan clickableSpan = new ClickableSpan() {

            @Override
            public void onClick(View view) {
                bus.post(event);
            }
        };

        text.setSpan(clickableSpan, 1/*skip the empty space*/, text.length(), Spannable.SPAN_EXCLUSIVE_EXCLUSIVE);
        transcriptTextView.append(text);
        transcriptTextView.setMovementMethod(LinkMovementMethod.getInstance());
    }


    /**
   * Task which wipes info message on UI thread
   */
  private final Runnable wipeInformationalMessage = new Runnable() {
    @Override
    public void run() {
      uiExecutor.execute(new Runnable() {
        @Override
        public void run() {
          setInformationalMessageImpl("");
        }
      });
    }
  };

  private final Runnable updateUserTyping = new Runnable() {
    @Override
    public void run() {
      stopTyping();
    }
  };

  ScheduledFuture<?> scheduledWipeInformationalMessage;
  ScheduledFuture<?> scheduledStopTypingMessage;

  private void showInfo(String text) {
    showInfoImpl(text, false);
  }

  private void showPermanentInfo(String text) {
    showInfoImpl(text, true);
  }

  private void showInfoImpl(String text, boolean permanent) {
    if (scheduledWipeInformationalMessage != null) {
      scheduledWipeInformationalMessage.cancel(false);
    }

    setInformationalMessageImpl(text);

    if (!permanent) {
      scheduledWipeInformationalMessage = timer.schedule(wipeInformationalMessage, 10, TimeUnit.SECONDS);
    }
  }

  private void setInformationalMessageImpl(final String text) {
    infoTextView.setText(text);
  }

  /**
   * First time the Close button is clicked, chat session is disconnected.
   * Subsequent click (or if chat session is finished) will close the activity.
   * @param item
   * @return
   */
  @Override
  public boolean onOptionsItemSelected(MenuItem item) {
    if (item.getItemId() == R.id.close) {
      if (!chatFinished) {
        chatFinished = true;
        controller.disconnectChat();
        NotificationManagerCompat.from(this).cancel(NID_CHAT_ACTIVY);
        this.finish();
      } else {
        finish();
      }
    }
    return true;
  }


  public void onEvent(FileUploadResponseEvent event){
      appendTranscriptInfo("File sent to Agent\n");
  }

  public void onEventMainThread(ChatResponseEvent event) {
    switch (event.chatRequestType) {
      case START:
        // handle start result
        break;
      case SEND:
        // handle send result
        break;
      case REFRESH:
        // handle refresh result
        break;
      case START_TYPING:
        break;
      case STOP_TYPING:
        break;
      case DISCONNECT:
        finishChat();
        break;
      default:
        break;
    }
  }

  private void finishChat() {
    chatFinished = true;
    updateSendButtonState();
    if (textWatcher != null) {
      sendEditText.removeTextChangedListener(textWatcher);
      textWatcher = null;
    }
    sendEditText.setText("");
    sendEditText.setEnabled(false);
    showPermanentInfo("Chat finished");
    NotificationManagerCompat.from(this).cancel(NID_CHAT_ACTIVY);
  }

  public void onEventMainThread(ChatTranscriptEvent event) {
    TranscriptEntry transcriptEntry = event.transcriptEntry;
    if (transcriptEntry.getChatEvent() == null) {
      Timber.e("Unknown ChatEvent encountered, see raw response for details.");
      return;
    }
    switch (transcriptEntry.getChatEvent()) {
      case PARTY_JOINED:
      case PARTY_LEFT:
        appendTranscriptInfo(transcriptEntry.getNickname() + " " + transcriptEntry.getText() + "\n");
        break;
      case TYPING_STARTED:
      case TYPING_STOPPED:
        showInfo(transcriptEntry.getNickname() + " " + transcriptEntry.getText());
        break;
      case MESSAGE:
      case PUSH_URL:
        showPermanentInfo("");
        appendTranscriptMessage(transcriptEntry.getNickname(), transcriptEntry.getText());
        break;
    }
  }

    public void onEventMainThread(ChatV2TranscriptEvent event) {
        ChatV2Response transcriptEntry = event.response;

        Timber.d("transcriptEntry Event type: " +transcriptEntry.getType());

        if (transcriptEntry.getType() == null) {
            Timber.e("Unknown V2 ChatEvent encountered");
            return;
        }

        switch (transcriptEntry.getType()) {
            case "Message":
                showPermanentInfo("");
                appendTranscriptMessage(transcriptEntry.getFrom().getNickname(),
                        transcriptEntry.getText());
                break;

            case "ParticipantJoined":
                appendTranscriptInfo(transcriptEntry.getFrom().getNickname() + " Joined\n");
                break;

            case "ParticipantLeft":
                appendTranscriptInfo(transcriptEntry.getFrom().getNickname() + " Left\n");
                transcriptTextView.setEnabled(false);
                break;

            case "TypingStarted":
                showInfo(transcriptEntry.getFrom().getNickname() + " typing ...." );
                break;

            case "TypingStopped":
                showInfo("");
                break;

            case "FileUploaded":
                if (!transcriptEntry.getFrom().getType().equalsIgnoreCase("Client")) {
                  appendTranscriptInfo(transcriptEntry.getFrom().getNickname() + " sent you a file");
                  displayDownloadConfirmationDialog(transcriptEntry, event.secureKey);
                }
                break;

            default:
                break;

        }
    }

  private void displayDownloadConfirmationDialog(final ChatV2Response transcriptEntry, final String secureKey) {
    AlertDialog.Builder builder = new AlertDialog.Builder(this);

    builder.setTitle("Do you want to download the file");
    builder.setMessage("Are you sure?");

    builder.setPositiveButton("YES", new DialogInterface.OnClickListener() {

      public void onClick(DialogInterface dialog, int which) {
        // Download the file now
        appendTranscriptInfo("\n");
        bus.post(new FileDownloadRequestEvent(transcriptEntry.getText(), secureKey));
        dialog.dismiss();
      }
    });

    builder.setNegativeButton("NO", new DialogInterface.OnClickListener() {

      @Override
      public void onClick(DialogInterface dialog, int which) {
        // add link in transcript to download later
        appendDownloadLink(new FileDownloadRequestEvent(transcriptEntry.getText(), secureKey));
        dialog.dismiss();
      }
    });

    AlertDialog alert = builder.create();
    alert.show();
  }

  @Override
  public void onBackPressed() {
    if (!chatFinished) {
        generateNotification();
    }
    //super.onBackPressed();
  }


  private void generateNotification() {
      Intent intentReturnToChat = new Intent(this, GenesysChatActivity.class);
      intentReturnToChat.setFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP | Intent.FLAG_ACTIVITY_SINGLE_TOP);
      PendingIntent pendingIntentReturn = PendingIntent.getActivity(
              this, 0, intentReturnToChat, PendingIntent.FLAG_ONE_SHOT);

      NotificationManagerCompat.from(this).notify(
              NID_CHAT_ACTIVY,
              new NotificationCompat.Builder(this)
                      .setSmallIcon(R.mipmap.ic_launcher_genesys)
                      .setOngoing(true)
                      .setContentTitle("GME")
                      .setContentIntent(pendingIntentReturn)
                      .setContentText(getResources().getString(R.string.chat_in_progress))
                      .setAutoCancel(true)
                      .setDefaults(Notification.DEFAULT_LIGHTS)
                      .build()
      );
  }


  public void onEventMainThread(ChatErrorEvent event) {
    Timber.e("Chat error encountered: %s", event.chatException);
    if ( event.chatException.getMessage().contains("disconnected")){
      chatFinished = true;
      NotificationManagerCompat.from(this).cancel(NID_CHAT_ACTIVY);
      finish();
    }
    Toast.makeText(this, event.chatException.getMessage(), Toast.LENGTH_SHORT).show();
    NotificationManagerCompat.from(this).cancel(NID_CHAT_ACTIVY);
  }

  public void transferFile(View view) {
    showFileChooser();
  }

}
