/*
 * Copyright (C) 2015 Genesys
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.genesys.gms.mobile.callback.demo.legacy.widgets;

import android.content.Context;
import android.content.res.TypedArray;
import android.preference.DialogPreference;
import android.util.AttributeSet;
import android.util.Log;
import android.view.View;
import android.widget.NumberPicker;
import com.genesys.gms.mobile.callback.demo.legacy.R;
import com.genesys.gms.mobile.callback.demo.legacy.util.TimeHelper;
import hugo.weaving.DebugLog;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Days;
import timber.log.Timber;

import java.util.Arrays;

/**
 * Custom DialogPreference, allowing for selection of both
 * Date and Time on a single line of NumberPickers
 */
public class DateTimePreference extends DialogPreference {
  // TODO: Externalize format of Date
  final private static String TAG = "DateTimePreference";

  private String currentDT;
  private NumberPicker datePicker;
  private NumberPicker hourPicker;
  private NumberPicker minutePicker;
  private NumberPicker periodPicker;
  private int maxDays = 28;
  final private static String[] defaultPeriods = {"AM", "PM"};
    /*
    final private static String DAY_OF_MONTH_FORMAT = "%a %b %d";
    final private static String ISO8601_FORMAT = "%FT%T.000Z";
    final private static String FRIENDLY_FORMAT = "%a %b %e %I:%M %p";
    */

  final private static String DEFAULT_VALUE = "0000-00-00T00:00:00.000Z";

  /** Custom number formatter forces 0-prefixed two digit numbers */
  private class TwoDigitFormatter implements NumberPicker.Formatter {
    public String format(int value) {
      return String.format("%02d", value);
    }
  }

  public DateTimePreference(Context context, AttributeSet attrs) {
    super(context, attrs);
    setDialogLayoutResource(R.layout.datetime_dialog);
    setPositiveButtonText(android.R.string.ok);
    setNegativeButtonText(android.R.string.cancel);

    TypedArray a = context.obtainStyledAttributes(attrs, R.styleable.DateTimePreference);
    final int N = a.getIndexCount();
    for (int i = 0; i < N; ++i) {
      int attr = a.getIndex(i);
      if (attr == R.styleable.DateTimePreference_maxDays) {
        maxDays = Math.max(1, a.getInt(attr, 28));
      }
    }
    a.recycle();
  }

  /** Bind available values into NumberPickers */
  @Override
  protected void onBindDialogView(View view) {
    super.onBindDialogView(view);

    // Initialize Picker elements
    // Note that dates will not be initialized
    datePicker = (NumberPicker) view.findViewById(R.id.datePicker);
    populateDates();
    datePicker.setMaxValue(maxDays - 1);
    datePicker.setMinValue(0);
    datePicker.setWrapSelectorWheel(false);
    datePicker.setDescendantFocusability(NumberPicker.FOCUS_BLOCK_DESCENDANTS);

    TwoDigitFormatter twoDigitFormatter = new TwoDigitFormatter();
    hourPicker = (NumberPicker) view.findViewById(R.id.hourPicker);
    hourPicker.setMaxValue(12);
    hourPicker.setMinValue(1);
    hourPicker.setFormatter(twoDigitFormatter);
    hourPicker.setDescendantFocusability(NumberPicker.FOCUS_BLOCK_DESCENDANTS);
    minutePicker = (NumberPicker) view.findViewById(R.id.minutePicker);
    minutePicker.setMaxValue(59);
    minutePicker.setMinValue(0);
    minutePicker.setFormatter(twoDigitFormatter);
    minutePicker.setDescendantFocusability(NumberPicker.FOCUS_BLOCK_DESCENDANTS);
    periodPicker = (NumberPicker) view.findViewById(R.id.periodPicker);
    periodPicker.setMaxValue(1);
    periodPicker.setMinValue(0);
    periodPicker.setWrapSelectorWheel(false);
    periodPicker.setDisplayedValues(defaultPeriods);
    periodPicker.setDescendantFocusability(NumberPicker.FOCUS_BLOCK_DESCENDANTS);

    setPickers();
  }

  /** Generate, format, and populate date strings into Date picker */
  public void populateDates() {
    String[] displayedDates = new String[maxDays];
    DateTime time = new DateTime();

    // Populate datePicker values based on maxDays setting
    displayedDates[0] = "Today";

    for (int i = 1; i < maxDays; ++i) {
      time = time.plusDays(1);
      displayedDates[i] = time.toString(TimeHelper.DAY_OF_MONTH_FORMAT);
    }

    Timber.d("displayedDates: %s", Arrays.toString(displayedDates));
    datePicker.setDisplayedValues(displayedDates);
  }

  /** Returns the number of days between two given DateTime inputs */
  private int getDayDifference(DateTime time1, DateTime time2) {
    return Days.daysBetween(time1.withTimeAtStartOfDay(), time2.withTimeAtStartOfDay()).getDays();
  }

  /** Extract selected DateTime value from pickers and convert to UTC format */
  private void readPickers() {
    int day = datePicker.getValue(); // Days from Today
    int hour = hourPicker.getValue();
    int minute = minutePicker.getValue();
    int period = periodPicker.getValue();
    hour %= 12;
    if (period == 1) // PM
    {
      hour += 12;
    }
    DateTime now = DateTime.now();
    DateTime time = now.plusDays(day).withTime(hour, minute, 0, 0);

    // If time before now is not allowed
    if (time.isBeforeNow()) {
      time = now;
    }

    currentDT = TimeHelper.serializeUTCTime(time);
  }

  /**
   * Given a DateTime in UTC format, populate values into pickers.
   * Note that this will default to current time if currentDT is null
   */
  private void setPickers() {
    // Set to currentDT
    DateTime time;
    DateTime today = DateTime.now();
    try {
      time = TimeHelper.parseISO8601DateTime(currentDT);
      time = time.withZone(DateTimeZone.getDefault());
    } catch (NullPointerException ex) {
      Timber.w("currentDT is null.");
      time = today;
    } catch (IllegalArgumentException ex) {
      Log.w(TAG, "currentDT is invalid: " + ex.getMessage());
      Timber.w(ex, "currentDT is invalid.");
      time = today;
    }

    int daysDiff = 0;

    Timber.d(
        "currentDT: %s, today: %s",
        TimeHelper.serializeUTCTime(time),
        TimeHelper.serializeUTCTime(DateTime.now())
    );
    // Only if time before now is not allowed
    if (time.isBefore(today)) {
      time = today;
    } else {
      daysDiff = getDayDifference(today, time);
      Timber.d("Day difference between restored and today: %d", daysDiff);
      // Previously selected time exceeds allowed range
      if (daysDiff > (maxDays - 1)) {
        time = today;
        daysDiff = 0;
      }
    }

    datePicker.setValue(daysDiff);
    if (time.getHourOfDay() >= 12) {
      hourPicker.setValue((time.getHourOfDay() - 12) == 0 ? 12 : (time.getHourOfDay() - 12));
      periodPicker.setValue(1);
    } else {
      hourPicker.setValue((time.getHourOfDay() == 0) ? 12 : time.getHourOfDay());
      periodPicker.setValue(0);
    }
    minutePicker.setValue(time.getMinuteOfHour());
  }

  private String formatDate() {
    return formatDate(null);
  }

  private String formatDate(DateTime time) {
    String result;
    if (time == null) {
      time = DateTime.now();
    }
    result = TimeHelper.serializeUTCTime(time);
    Timber.d("formatDate: %s", result);
    return result;
  }

  /** Extracts DateTime value from Pickers if dialog was closed */
  @Override
  @DebugLog
  protected void onDialogClosed(boolean positiveResult) {
    super.onDialogClosed(positiveResult);
    if (positiveResult) {
      readPickers();
      if (callChangeListener(currentDT)) {
        Timber.d("callChangeListener returned true.");
        persistString(currentDT);
        setSummary(TimeHelper.toFriendlyString(currentDT));
      }
    }
  }

  /** Initialize DateTimePreference if persisted value is available */
  @Override
  @DebugLog
  protected void onSetInitialValue(boolean restorePersistedValue, Object defaultValue) {
    if (restorePersistedValue) {
      currentDT = this.getPersistedString(DEFAULT_VALUE);
    } else {
      if (defaultValue == null) {
        currentDT = formatDate();
      } else {
        currentDT = (String) defaultValue;
      }
      persistString(currentDT);
    }
  }

  @Override
  protected Object onGetDefaultValue(TypedArray a, int index) {
    return a.getString(index);
  }
}
