/*
 * Copyright (C) 2015 Genesys
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.genesys.gms.mobile.callback.demo.legacy.data;

import android.app.Application;
import android.content.Context;
import android.content.SharedPreferences;
import android.preference.PreferenceManager;

import com.genesys.gms.mobile.callback.demo.legacy.common.ForApplication;
import com.genesys.gms.mobile.callback.demo.legacy.data.api.ApiModule;
import com.genesys.gms.mobile.callback.demo.legacy.data.api.pojo.TranscriptEntry;
import com.genesys.gms.mobile.callback.demo.legacy.data.gson.DateTimeTypeAdapter;
import com.genesys.gms.mobile.callback.demo.legacy.data.gson.TranscriptEntryTypeAdapter;
import com.genesys.gms.mobile.callback.demo.legacy.util.Globals;
import com.google.gson.FieldNamingPolicy;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.squareup.okhttp.Cache;
import com.squareup.okhttp.OkHttpClient;

import org.cometd.client.transport.ClientTransport;
import org.cometd.websocket.client.JettyWebSocketTransport;
import org.eclipse.jetty.client.HttpClient;
import org.eclipse.jetty.websocket.client.WebSocketClient;
import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import java.io.File;
import java.io.IOException;

import javax.inject.Singleton;
import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSession;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;

import dagger.Module;
import dagger.Provides;

/**
 * Dagger module which hooks up all the API managers and utilities
 */
@Module(
    includes = ApiModule.class,
    complete = false,
    library = true
)
public class DataModule {
  static final int DISK_CACHE_SIZE = 50 * 1024 * 1024; // 50MB
  static final String UTC_DATE_FORMAT = "yyyy'-'MM'-'dd'T'HH':'mm':'ss'.'SSS'Z'";

  @Provides
  @Singleton
  OkHttpClient provideOkHttpClient(Application app) {
    return createOkHttpClient(app);
  }

  @Provides
  @Singleton
  HttpClient provideJettyHttpClient() {
    return createJettyHttpClient();
  }

  @Provides
  @Singleton
  ClientTransport provideClientTransport() {return getWebSocketTransport();}

  @Provides
  @Singleton
  DateTimeFormatter provideDateTimeFormatter() {
    return DateTimeFormat.forPattern(UTC_DATE_FORMAT).withZoneUTC();
  }

  @Provides
  @Singleton
  Gson provideGson() {
    return new GsonBuilder()
        .setFieldNamingPolicy(FieldNamingPolicy.IDENTITY)
        .setDateFormat(UTC_DATE_FORMAT)
        .registerTypeAdapter(DateTime.class, new DateTimeTypeAdapter())
        .registerTypeAdapter(TranscriptEntry.class, new TranscriptEntryTypeAdapter())
        .setPrettyPrinting()
        .create();
  }

  @Provides
  @Singleton
  SharedPreferences provideSharedPreferences(@ForApplication Context context) {
    return PreferenceManager.getDefaultSharedPreferences(context);
  }

  static OkHttpClient createOkHttpClient(Application app) {
    OkHttpClient client = getUnsafeOkHttpClient();

    // Install an HTTP cache in the application cache directory.
    try {
      File cacheDir = new File(app.getCacheDir(), "http");
      Cache cache = new Cache(cacheDir, DISK_CACHE_SIZE);
      client.setCache(cache);
    } catch (IOException e) {
      //Timber.e(e, "Unable to install disk cache.");
    }

    return client;
  }

  static HttpClient createJettyHttpClient() {
    HttpClient httpClient = new HttpClient();
    httpClient.setConnectTimeout(Globals.CONNECT_TIMEOUT);
    /* SetThreadPool is deprecated, Threadpool is created internally and random name is set */
    try {
      httpClient.start();
    } catch (Exception e) {
      throw new RuntimeException(e);
    }
    return httpClient;
  }


  private static OkHttpClient getUnsafeOkHttpClient() {
    try {
      // Create a trust manager that does not validate certificate chains
      final TrustManager[] trustAllCerts = new TrustManager[] {
              new X509TrustManager() {
                @Override
                public void checkClientTrusted(java.security.cert.X509Certificate[] chain, String authType)  {
                }

                @Override
                public void checkServerTrusted(java.security.cert.X509Certificate[] chain, String authType)  {
                }

                @Override
                public java.security.cert.X509Certificate[] getAcceptedIssuers() {
                  return null;
                }
              }
      };

      // Install the all-trusting trust manager
      final SSLContext sslContext = SSLContext.getInstance("SSL");
      sslContext.init(null, trustAllCerts, new java.security.SecureRandom());
      // Create an ssl socket factory with our all-trusting manager
      final SSLSocketFactory sslSocketFactory = sslContext.getSocketFactory();

      OkHttpClient okHttpClient = new OkHttpClient();
      okHttpClient.setSslSocketFactory(sslSocketFactory);
      okHttpClient.setHostnameVerifier(new HostnameVerifier() {
        @Override
        public boolean verify(String hostname, SSLSession session) {
          return true;
        }
      });

      return okHttpClient;
    } catch (Exception e) {
      throw new RuntimeException(e);
    }
  }

    private static ClientTransport getWebSocketTransport() {
      try {
          WebSocketClient webSocketClient = new WebSocketClient();
          webSocketClient.start();
          return new JettyWebSocketTransport(null, null, webSocketClient);
      }catch(Exception e) {
          throw new RuntimeException(e);
      }
    }
}

