﻿// Copyright © 2014 Genesys. All Rights Reserved.
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Genesyslab.Desktop.Infrastructure.DependencyInjection;
using Genesyslab.Desktop.Modules.Core.SDK.Protocol;
using Genesyslab.Platform.ApplicationBlocks.Commons.Broker;
using Genesyslab.Platform.ApplicationBlocks.Commons.Protocols;
using Genesyslab.Platform.Commons.Protocols;
using Genesyslab.Platform.Commons.Logging;
using Genesyslab.Platform.Voice.Protocols.TServer.Requests.Dn;
using Genesyslab.Platform.Voice.Protocols;
using Genesyslab.Platform.Voice.Protocols.TServer.Events;
using Genesyslab.Platform.Commons.Collections;
using Genesyslab.Desktop.Modules.Gms.CallbackInvitation.Generic;
using System.Windows;
using System.Windows.Threading;
using Genesyslab.Desktop.Infrastructure.Commands;

namespace Genesyslab.Desktop.Modules.Gms.CallbackInvitation.CallbackInvitation {
    class UserEventListener: IUserEventListener {
        private readonly IObjectContainer container = null;
        private readonly ICfgReader config = null;
        private readonly ILogger log = null;

        private TServerProtocol protocol = null;
        private String _agentDN = "";
        private Boolean protocolOpened = false;

        private const String clientName = "Workspace_CallbackInvitation";
        private const int eventUserEventId = 96;
        private const String source = "ORS";    // Source of the event
        private const String eventCustomName = "CallbackInvitationEvent";
        private const String eventExtensionSourceKey = "source";
        private const String eventExtensionEventKey = "event";
        private const String eventExtensionTimeoutKey = "invitation-timeout";
        private const String eventUserDisplayDataKey = "display-data";
        private const String eventUserResponseOptionsKey = "response-options";
        private const int defaultTimeout = 30;

        public UserEventListener(IObjectContainer container) {
            this.container = container;
            this.config = this.container.Resolve<ICfgReader>();
            this.log = this.container.Resolve<ILogger>().CreateChildLogger("UserEventListener");
        }

        private void EventHandlerProtocolOpened(object sender, EventArgs e) {
            protocolOpened = true;
        }

        private void EventHandlerProtocolClosed(object sender, EventArgs e) {
            protocolOpened = false;
        }

        private void EventHandlerTServerReceived(object sender, EventArgs e) {
            MessageEventArgs args = e as MessageEventArgs;
            if (args != null) {
                if (args.Message.Id.Equals(eventUserEventId)) {
                    EventUserEvent userEventData = args.Message as EventUserEvent;

                    // Determine if to continue parsing of this UserEvent (depends on AttributeExtensions and AttributeUserData set)
                    Boolean continueParsing = (userEventData.Extensions != null) ?
                        ((userEventData.Extensions.ContainsKey(eventExtensionSourceKey)) && (userEventData.Extensions.ContainsKey(eventExtensionEventKey))) :
                        false;
                    continueParsing = (continueParsing) ?
                        ((userEventData.Extensions[eventExtensionSourceKey].Equals(source)) && (userEventData.Extensions[eventExtensionEventKey].Equals(eventCustomName))) :
                        false;
                    continueParsing = ((continueParsing) && (userEventData.UserData != null)) ?
                        ((userEventData.UserData.ContainsKey(eventUserDisplayDataKey)) && (userEventData.UserData.ContainsKey(eventUserResponseOptionsKey))) :
                        false;

                    if (continueParsing) {
                        this.log.Debug(String.Format("{0} Processing event [{1}] with contents: {2}", GConst.LOG_PREFIX,
                                args.Message.Name, args.Message.ToString()));

                        // Retrieve the relevant information of the UserEvent
                        int invitationTimeout = (userEventData.Extensions.ContainsKey(eventExtensionTimeoutKey)) ?
                                Int16.Parse(userEventData.Extensions.GetAsString(eventExtensionTimeoutKey)) : defaultTimeout;
                        KeyValueCollection displayData = userEventData.UserData.GetAsKeyValueCollection(eventUserDisplayDataKey);
                        KeyValueCollection responseOptions = userEventData.UserData.GetAsKeyValueCollection(eventUserResponseOptionsKey);

                        // The InvitationDialog can only be invoked on the main thread
                        Application.Current.Dispatcher.Invoke((Action)delegate {
                            InvitationDialog invitationDialog = new InvitationDialog(this.container, displayData, responseOptions, invitationTimeout);

                            bool? dialogResult = invitationDialog.ShowDialog();
                        });
                    }
                    else {
                        this.log.Debug(String.Format("{0} Ignoring non-relevant UserEvent (doesn't have the necessary AttributeExtensions / AttributeUserData)", GConst.LOG_PREFIX));
                    }
                }
                else {
                    this.log.Debug(String.Format("{0} Ignoring event [{1}]", GConst.LOG_PREFIX, args.Message.Name));
                }
            }
        }

        public void SetupConnection() {
            this.log.Debug (String.Format ("{0} Setting up connection to T-Server", GConst.LOG_PREFIX));

            if (!String.IsNullOrEmpty(this._agentDN)) {
                // Open the connection to T-Server
                EventReceivingBrokerService eventBroker = new EventReceivingBrokerService();
                protocol = new TServerProtocol(this.config.EndpointTServer);
                protocol.ClientName = clientName;
                protocol.Opened += new EventHandler(EventHandlerProtocolOpened);
                protocol.Closed += new EventHandler(EventHandlerProtocolClosed);

                // Subscribe an event handler for all TServer events
                protocol.Received += new EventHandler(EventHandlerTServerReceived);

                protocol.Open();

                this.log.Debug(String.Format("{0} Connection from custom client [{1}] to T-Server is opened",
                        GConst.LOG_PREFIX, clientName));

                // Register the DN, so that events for this DN will be received
                RequestRegisterAddress req = RequestRegisterAddress.Create();
                req.ThisDN = this._agentDN;
                req.RegisterMode = Genesyslab.Platform.Voice.Protocols.TServer.RegisterMode.ModeShare;
                req.ControlMode = Genesyslab.Platform.Voice.Protocols.TServer.ControlMode.RegisterDefault;
                req.AddressType = Genesyslab.Platform.Voice.Protocols.TServer.AddressType.DN;
                protocol.Send(req);

                this.log.Debug(String.Format("{0} Registered for events sent to DN [{1}]", GConst.LOG_PREFIX, this._agentDN));
            }
            else
                throw new Exception("T-Server connection cannot be set up: no Agent DN has been set");
        }

        public void CloseConnection() {
            if ((protocol != null) && (this.protocolOpened)) {
                protocol.Close();
                this.log.Debug(String.Format("{0} Connection from custom client [{1}] to T-Server closed",
                        GConst.LOG_PREFIX, clientName));
            }
            else {
                this.log.Debug(String.Format("{0} Connection from custom client [{1}] to T-Server doesn't need to be closed",
                        GConst.LOG_PREFIX, clientName));
            }
        }

        public String AgentDN {
            get { return this._agentDN; }
            set { if (this._agentDN != value) { this._agentDN = value; } }
        }

        /// <summary>
        /// This delegate allows to go to the main thread.
        /// </summary>
        delegate bool ExecuteDelegate(IDictionary<string, object> parameters, IProgressUpdater progressUpdater);
    }
}
