// Copyright  2015 Genesys. All Rights Reserved.
using System;
using System.Collections.Generic;
using Genesyslab.Desktop.Infrastructure;
using Genesyslab.Desktop.Infrastructure.Commands;
using Genesyslab.Desktop.Infrastructure.DependencyInjection;
using Genesyslab.Desktop.Infrastructure.ViewManager;
using Genesyslab.Desktop.Modules.Core.Model.Interactions;
using Genesyslab.Desktop.Modules.Gms.CallbackInvitation.CustomCommands;
using Genesyslab.Desktop.Modules.Gms.CallbackInvitation.Generic;
using Genesyslab.Platform.Commons.Logging;
using Genesyslab.Desktop.Modules.Gms.CallbackInvitation.CallbackInvitation;

namespace Genesyslab.Desktop.Modules.Gms.CallbackInvitation
{
    public struct CommandActivatorData {
        public String name;
        public String chain;
        public String afterCommand;
        public Type type;
    }

	/// <summary>
	/// This class is the main module through which all forms and buttons are registered with Interaction Workspace
	/// </summary>
	public class CallbackInvitationModule : IModule
	{
        readonly IObjectContainer container;
		readonly IViewManager viewManager;
		readonly ICommandManager commandManager;
        private IInteractionManager interactionManager;
        private readonly ILogger log = null;        

		/// <summary>
		/// Initializes a new instance of the ScreenpopModule class.
		/// </summary>
		/// <param name="container">The container.</param>
		/// <param name="viewManager">The view manager.</param>
		/// <param name="commandManager">The command manager.</param>
        public CallbackInvitationModule(IObjectContainer container, IViewManager viewManager, ICommandManager commandManager)
		{
			this.container = container;
			this.viewManager = viewManager;
			this.commandManager = commandManager;

            this.interactionManager = container.Resolve<IInteractionManager>();

            // Initialize logging
            log = container.Resolve<ILogger>();
            log = log.CreateChildLogger("Screenpop");
            
		}

		/// <summary>
		/// Initializes the module
		/// </summary>
		public void Initialize()
		{
            log.Info(String.Format("{0} Initialization started...", GConst.LOG_PREFIX));

            // Read configuration and register in (global) container
            CfgReader config = new CfgReader(container);
            container.RegisterInstance<ICfgReader>(config);

            // Register instance of the UserEventListener, however, it cannot be setup before the agent has logged on.
            // First then, the place name is known.
            UserEventListener userEventListener = new UserEventListener(container);
            container.RegisterInstance<IUserEventListener>(userEventListener);


            // Registering custom commands
            List<CommandActivatorData> customCommandList = new List<CommandActivatorData>();
            customCommandList.Add(
                    new CommandActivatorData() { chain = "ApplicationClose", afterCommand = "IsPossibleToClose ", type = typeof(BeforeApplicationCloseCommand), name = "BeforeApplicationCloseCommand" });
            customCommandList.Add(
                    new CommandActivatorData() { chain = "MediaVoiceLogOn", afterCommand = "LogOn ", type = typeof(AfterMediaVoiceLogOnCommand), name = "AfterMediaVoiceLogOnCommand" });

            foreach (CommandActivatorData thisCommand in customCommandList) {
                this.log.Debug(String.Format("{0} Inserting custom command [{1}] in command chain", GConst.LOG_PREFIX, thisCommand.name));

                IList<CommandActivator> insertedList = new List<CommandActivator>();
                insertedList.Add(new CommandActivator() { CommandType = thisCommand.type, Name = thisCommand.name });

                this.commandManager.InsertCommandToChainOfCommandAfter(thisCommand.chain, thisCommand.afterCommand, insertedList);
            }

            log.Info(String.Format("{0} Initialization completed...", GConst.LOG_PREFIX));
		}
	}
}
