﻿using System;
using System.Collections.Generic;
using System.Linq;
using Genesyslab.Desktop.Infrastructure;
using Genesyslab.Desktop.Infrastructure.Commands;
using Genesyslab.Desktop.Modules.Core.Model.Agents;
using Genesyslab.Desktop.Modules.Core.Model.BusinessAttributes;
using Genesyslab.Desktop.Modules.Core.Model.Interactions;
using Genesyslab.Desktop.Modules.Core.Utils.Context;
using Genesyslab.Desktop.Modules.OpenMedia;
using Genesyslab.Desktop.Modules.OpenMedia.Model.Agents;
using Genesyslab.Desktop.Modules.OpenMedia.Model.Interactions;
using Genesyslab.Enterprise.Commons.Collections;
using Genesyslab.Enterprise.Extensions;
using Genesyslab.Enterprise.Model.Channel;
using Genesyslab.Enterprise.Model.Device;
using Genesyslab.Enterprise.Model.Envelope;
using Genesyslab.Enterprise.Model.Interaction;
using Genesyslab.Enterprise.Model.ServiceModel;
using Genesyslab.Enterprise.Services;
using Genesyslab.Platform.Commons.Logging;
using Microsoft.Practices.Unity;
using Tomers.WPF.Localization;

namespace Genesyslab.Desktop.Modules.InteractionManagementExtensionSample
{

    class CustomOpenMediaCommand : IElementOfCommand
    {
        protected readonly IUnityContainer container;
        protected ILogger log;
        protected IOpenMediaService openMediaService;
        protected IChannelService channelService;
        protected const int timeout = 10000;

        protected IAgentMultimedia agentMultimedia;
        protected IAgent agent;

        public CustomOpenMediaCommand(IUnityContainer container)
        {
            this.container = container;
            this.log = container.Resolve<ILogger>();
            this.log = log.CreateChildLogger("CustomOpenMediaCommand");

            IEnterpriseServiceProvider enterpriseServiceProvider = container.Resolve<IEnterpriseServiceProvider>();
            this.openMediaService = enterpriseServiceProvider.Resolve<IOpenMediaService>("openmediaService");
            this.channelService = enterpriseServiceProvider.Resolve<IChannelService>("channelService");
            this.agent = container.Resolve<IAgent>();
            this.agentMultimedia = container.Resolve<IAgentMultimedia>();
        }

        public virtual string Name { get; set; }


        #region IElementOfCommand Members

        public virtual bool Execute(IDictionary<string, object> parameters, IProgressUpdater progressUpdater)
        {
            return false;
        }

        #endregion

        public void PutInteractionBackToOriginalLocation(IOpenMediaInteraction interactionEsdk)
        {
            try
            {
                IEnvelope<IOpenMediaInteraction> putBackResult = openMediaService.PlaceInQueue(interactionEsdk, "__BACK__", null, null);
                if (putBackResult == null)
                    log.WarnFormat("PutInteractionBackToOriginalLocation failed, {0}", interactionEsdk);

                if ((putBackResult.Header != null) && (putBackResult.Header.CurrentContext != null) && (putBackResult.Header.CurrentContext.msg != null))
                {
                    if ((putBackResult.Header.CurrentContext.msg.Name == "EventError") || (putBackResult.Header.CurrentContext.msg.Name == "ErrorEvent"))
                        log.WarnFormat("PutInteractionBackToOriginalLocation returns an error,{0}", interactionEsdk);
                }
            }
            catch (Exception exp)
            {
                log.WarnFormat("PutInteractionBackToOriginalLocation, Exception " + interactionEsdk, exp);
            }
        }
    }


    [ElementOfCommand(parameters = @"CommandParameter: System.String|
                MarkDoneReason: System.String")]
    class InteractionManagementMarkDoneWithReasonCommand : CustomOpenMediaCommand
    {
        public InteractionManagementMarkDoneWithReasonCommand(IUnityContainer container) : base(container) { }

        public override bool Execute(IDictionary<string, object> parameters, IProgressUpdater progressUpdater)
        {
            log.Info("InteractionManagementMarkDoneWithReasonCommand");

            log.Info("InteractionManagementMarkDoneWithReasonCommand");
            string interactionId = parameters.TryGetValue("CommandParameter") as string;
            if (string.IsNullOrEmpty(interactionId))
            {
                log.Error("InteractionManagementMarkDoneWithReasonCommand : interactionId in CommandParameter is null or empty ");

                return true;
            }

            string reason = parameters.TryGetValue("MarkDoneReason") as string;
            if (string.IsNullOrEmpty(reason))
            {
                log.Error("InteractionManagementMarkDoneWithReasonCommand : reason in CommandParameter is null or empty ");

                return true;
            }

            IDevice multimediaPlace = null;
            if (agentMultimedia != null && agentMultimedia.DevicePlace != null)
                multimediaPlace = agentMultimedia.DevicePlace;
            else
            {
                if (agent != null && agent.Place != null)
                {
                    foreach (IMedia media in agent.Place.ListOfMedia)
                    {
                        IMediaOpenMedia openMedia = media as IMediaOpenMedia;
                        if (openMedia != null)
                        {
                            multimediaPlace = openMedia.AgentMultimedia.DevicePlace;
                            break;
                        }
                    }
                }
            }
            if (multimediaPlace == null)
            {
                string message = "Can't pull interaction and mark done it with reason. There is no place for the agent";
                log.Error(message);
                MultipleInteractionsCommandContext.SetSingleInteractionCommandContextFailure(container, parameters, message);

                return true;
            }

            IContextManager contextManager = container.Resolve<IContextManager>();
            string correlatorData = null;
            try
            {
                // Get Interaction from workbin or Queue with correlator data to prevent it from opening
                // Next step will be to SetReason and then to StopProcessing it... still with correlator data to filter events...
                correlatorData = contextManager.AddContext(correlatorData, "SilentWorkflowContextSample.NextCommand", "SetReasonStopProcessing");
                correlatorData = contextManager.AddContext(correlatorData, "MarkDoneReason", reason);
                //correlatorData = contextManager.AddContext(correlatorData, "WorkbinsView", workbinsView);

                log.Debug("[.SilentWorkflow.] InteractionManagementMarkDoneWithReasonCommand :   correlatorData = CustomSilentWorkflowContext.NextCommand = SetReasonStopProcessing, MarkDoneReason = " + reason);

                correlatorData = MultipleInteractionsCommandContext.SaveMultipleInteractionsCommandContextIdWithCorrelatorData(contextManager, parameters, correlatorData);

                IClientChannel channel = channelService.GetChannel(multimediaPlace);
                IEnvelope<IOpenMediaInteraction> env = openMediaService.Pull(channel, multimediaPlace, interactionId, null, null, 5000, correlatorData);

                if ((env != null) && (env.Header != null) && (env.Header.CurrentContext != null) && (env.Header.CurrentContext.ContextState == ContextStateType.Error))
                {
                    // Pull Failed
                    string message = LanguageDictionary.GetDictionary(LanguageContext.Instance.Culture).Translate("InteractionManagement.PullFailed", "Text", "Pull failed", typeof(string)) as string;
                    if (string.IsNullOrEmpty(env.Header.Message))
                        message += ".";
                    else
                        message += " : " + env.Header.Message;
                    log.Error("[.SilentWorkflow.] InteractionManagementMarkDoneWithReasonCommand id = " + interactionId + ", " + message);
                    MultipleInteractionsCommandContext.SetSingleInteractionCommandContextFailure(contextManager, correlatorData, parameters, message);

                    return true;
                }
            }
            catch (Exception exp)
            {
                string message = "InteractionManagementMarkDoneWithReasonCommand id = " + interactionId + ", Exception ";
                log.Error(message, exp);
                MultipleInteractionsCommandContext.SetSingleInteractionCommandContextFailure(contextManager, correlatorData, parameters, message);

                return true;
            }

            return false;
        }
    }

    [ElementOfCommand(parameters = @"CommandParameter: Genesyslab.Enterprise.Model.Interaction.IOpenMediaInteraction|
                MarkDoneReason: System.String")]
    class InteractionOpenMediaEsdkSetReasonCommand : CustomOpenMediaCommand
    {
        public InteractionOpenMediaEsdkSetReasonCommand(IUnityContainer container) : base(container) { }

        public override bool Execute(IDictionary<string, object> parameters, IProgressUpdater progressUpdater)
        {
            log.Info("InteractionOpenMediaEsdkSetReasonCommand");

            IOpenMediaInteraction interactionEsdk = parameters.TryGetValue("CommandParameter") as IOpenMediaInteraction;
            if (interactionEsdk == null)
            {
                log.Error("InteractionOpenMediaEsdkSetReasonCommand - Can't set reason, because interactionEsdk is null");

                return true;
            }

            string reason = parameters.TryGetValue("MarkDoneReason") as string;
            if (string.IsNullOrEmpty(reason))
            {
                log.Error("InteractionOpenMediaEsdkSetReasonCommand - Reason is null or empty");

                return true;
            }

            try
            {
                KeyValueCollection data = new KeyValueCollection();
                data["Reason"] = reason;

                openMediaService.SetAttachedData(interactionEsdk, data, 5000, null);
            }
            catch (Exception exp)
            {
                // An error occurs during SetAttachedData... command returns true : the chain or command is Stopped
                // >> Put the interaction back to its original location (from which it has been pulled)
                PutInteractionBackToOriginalLocation(interactionEsdk);

                string message = "InteractionOpenMediaEsdkSetReasonCommand, Exception ";
                log.Error(message, exp);
                MultipleInteractionsCommandContext.SetSingleInteractionCommandContextFailure(container, parameters, message);

                return true;
            }

            return false;
        }
    }


    [ElementOfCommand(parameters = @"CommandParameter: Genesyslab.Enterprise.Model.Interaction.IOpenMediaInteraction")]
    class InteractionOpenMediaEsdkStopProcessingCommand : CustomOpenMediaCommand
    {
        public InteractionOpenMediaEsdkStopProcessingCommand(IUnityContainer container) : base(container) { }

        public override bool Execute(IDictionary<string, object> parameters, IProgressUpdater progressUpdater)
        {
            log.Info("InteractionOpenMediaEsdkStopProcessingCommand");

            IOpenMediaInteraction esdkOpenMedia = parameters.TryGetValue("CommandParameter") as IOpenMediaInteraction;
            if (esdkOpenMedia == null)
            {
                log.Error("InteractionOpenMediaEsdkStopProcessingCommand esdkOpenMedia is null");

                return true;
            }

            if (parameters.ContainsKey("Workflow"))
            {
                if ((parameters["Workflow"] as string) == "No")
                {
                    log.Info("InteractionOpenMediaEsdkStopProcessingCommand, find no workflow");
                    MultipleInteractionsCommandContext.SetSingleInteractionCommandContextSuccess(container, parameters);

                    return false;
                }
            }

            try
            {
                if (!esdkOpenMedia.IsInWorkflow)
                {
                    log.Debug("InteractionOpenMediaEsdkStopProcessingCommand, interaction is already not in workflow anymore");
                    MultipleInteractionsCommandContext.SetSingleInteractionCommandContextSuccess(container, parameters);

                    return false;
                }

                bool stopProcessing = true;
                string firstQueue = null;

                ICollection<string> inputQueues = esdkOpenMedia.InputQueues;
                if (inputQueues != null)
                {
                    if (inputQueues.Count > 0)
                    {
                        firstQueue = inputQueues.First();

                        if (!"__STOP__".Equals(firstQueue))
                        {
                            stopProcessing = false;
                        }
                    }
                }

                if (stopProcessing)
                {
                    IChainOfCommand stopProcessingUCSCommand = null;
                    IDictionary<string, object> stopProcessingUCSParameters = null;

                    stopProcessingUCSCommand = container.Resolve<ICommandManager>().GetChainOfCommandByName("InteractionOpenMediaUCSStopProcessing");

                    if (stopProcessingUCSCommand != null)
                    {
                        stopProcessingUCSParameters = new Dictionary<string, object>();
                        stopProcessingUCSParameters["CommandParameter"] = esdkOpenMedia.Id;

                        KeyValueCollection userData = openMediaService.GetAttachedData(esdkOpenMedia);

                        KeyValueCollection attachedDataInformation = parameters.TryGetValue("AttachedDataInformation") as KeyValueCollection;
                        if (attachedDataInformation == null)
                            log.Debug("InteractionOpenMediaEsdkStopProcessingCommand attachedDataInformation is null");
                        IDispositionCode dispositionCode = parameters.TryGetValue("DispositionCode") as IDispositionCode;
                        if (dispositionCode == null)
                            log.Debug("InteractionOpenMediaEsdkStopProcessingCommand dispositionCode is null");

                        if (OpenMediaOptions.Default.InteractionAttachedDataInformationIsEnable(esdkOpenMedia.IdType.SubMediaType))
                        {
                            if (dispositionCode != null)
                            {
                                if (attachedDataInformation == null)
                                    attachedDataInformation = new KeyValueCollection();

                                attachedDataInformation.Add("DispositionCode.Key", dispositionCode.CodeKeyName);
                                attachedDataInformation.Add("DispositionCode.Label", dispositionCode.Label);

                                foreach (IDispositionCodeValue dispositionCodeValue in dispositionCode.ListOfDispositionCodeValue)
                                {
                                    if (!string.IsNullOrEmpty(dispositionCodeValue.Name))
                                        attachedDataInformation.Add("DispositionCodeValues." + dispositionCodeValue.Name, dispositionCodeValue.DisplayName);
                                }
                            }

                            if (attachedDataInformation != null)
                            {
                                if (userData == null)
                                {
                                    userData = new KeyValueCollection();
                                }
                                userData[InteractionUserData.IWAttachedDataInformation] = attachedDataInformation;
                            }
                        }

                        if (userData != null)
                        {
                            stopProcessingUCSParameters["UserData"] = userData;
                        }
                    }

                    log.DebugFormat("InteractionOpenMediaEsdkStopProcessingCommand, openMediaService.StopProcessing {0}", esdkOpenMedia.Id);
                    openMediaService.StopProcessing(esdkOpenMedia,
                        parameters.TryGetValue("Reason") as KeyValueCollection,
                        parameters.TryGetValue("Extensions") as KeyValueCollection);

                    if (stopProcessingUCSCommand != null)
                    {
                        stopProcessingUCSCommand.BeginExecute(stopProcessingUCSParameters,
                             delegate(IAsyncResult ar)
                             {
                                 stopProcessingUCSCommand.EndExecute(ar);
                             }, null);
                    }
                }
                else
                {
                    log.DebugFormat("InteractionOpenMediaEsdkStopProcessingCommand, openMediaService.PlaceInQueue {0} {1}", firstQueue, esdkOpenMedia.Id);
                    openMediaService.PlaceInQueue(esdkOpenMedia,
                        firstQueue,
                        parameters.TryGetValue("Reason") as KeyValueCollection,
                        parameters.TryGetValue("Extensions") as KeyValueCollection);
                }
            }
            catch (Exception exp)
            {
                // An error occurs during PlaceInQueue... command returns true : the chain or command is Stopped
                // >> Put the interaction back to its original location (from which it has been pulled)
                PutInteractionBackToOriginalLocation(esdkOpenMedia);

                string message = "InteractionOpenMediaEsdkStopProcessingCommand StopProcessing or PlaceInQueue, Exception " + esdkOpenMedia;
                log.Error(message, exp);
                MultipleInteractionsCommandContext.SetSingleInteractionCommandContextFailure(container, parameters, message);

                return true;
            }

            MultipleInteractionsCommandContext.SetSingleInteractionCommandContextSuccess(container, parameters);

            return false;
        }
    }

}
