/*=====================================================================
Any authorized distribution of any copy of this code (including any related documentation) must reproduce the following restrictions, disclaimer and copyright notice:
The Genesys name, trademarks and/or logo(s) of Genesys shall not be used to name (even as a part of another name), endorse and/or promote products derived from this code without prior written permission from Genesys Telecommunications Laboratories, Inc.
The use, copy, and/or distribution of this code is subject to the terms of the Genesys Developer License Agreement.  This code shall not be used, copied, and/or distributed under any other license agreement.
THIS CODE IS PROVIDED BY GENESYS TELECOMMUNICATIONS LABORATORIES, INC. (GENESYS) AS IS WITHOUT ANY WARRANTY OF ANY KIND. GENESYS HEREBY DISCLAIMS ALL EXPRESS, IMPLIED, OR STATUTORY CONDITIONS, REPRESENTATIONS AND WARRANTIES WITH RESPECT TO THIS CODE (OR ANY PART THEREOF), INCLUDING, BUT NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT. GENESYS AND ITS SUPPLIERS SHALL NOT BE LIABLE FOR ANY DAMAGE SUFFERED AS A RESULT OF USING THIS CODE. IN NO EVENT SHALL GENESYS AND ITS SUPPLIERS BE LIABLE FOR ANY DIRECT, INDIRECT, CONSEQUENTIAL, ECONOMIC, INCIDENTAL, OR SPECIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, ANY LOST REVENUES OR PROFITS).
Copyright &copy; 2006 Genesys Telecommunications Laboratories, Inc. All rights reserved.

=====================================================================*/
package agentserver;

import com.genesyslab.ail.Agent;
import com.genesyslab.ail.AilFactory;
import com.genesyslab.ail.Dn;
import com.genesyslab.ail.IdObject;
import com.genesyslab.ail.PersonSummary;
import com.genesyslab.ail.Place;
import com.genesyslab.ail.monitor.MonitorEvent;
import com.genesyslab.ail.monitor.MonitorEventAgentStatus;
import com.genesyslab.ail.applicationblocks.connector.ApplicationParameters;
import com.genesyslab.ail.exception.RequestFailedException;
import com.genesyslab.ail.monitor.Monitor;
import com.genesyslab.ail.monitor.MonitorListener;

import java.lang.String;
import java.lang.Integer;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Vector;

/**
 * <p>Implements a server that:</p>
 * <ul>
 * <li>Monitors stat events to get agent status changes.</li>
 * <li>Gets a list of default place to be used when a login action is requested.</li>
 * </ul>
 * 
 * @version 7.5.001.00
 */

public class AgentServer {
	/** Component to connect Genesys Servers. */
	public Connector connector;
	/** Application Block to pass connnection data. */
	public ApplicationParameters serverData;
	/** Specifies whether the agent Server application is connected to Genesys Framework. */ 
	public String connected = "not connected";
	/** Factory. */
	public AilFactory factory;
	/** Listener for agent status changes. */
	public AgentMonitorListener monitorListener;
	
	/** Agent status data. */
	public HashMap statusMap;
	Vector listenerVector;
	Vector placeVector;
	/** Indicates whether are available. */
	boolean dataAvailable = false;
	/** Indicates whether stat service is available. Without a connection to stat service, no stat are available and Agent Server cannot get events about statuses. */
	boolean isStatServiceAvailable = false;

	/**
	 * Constructor.
	 * <br>Creates the Agent Server and connects to Genesys Servers.
	 * <br>To connect, this class uses a <code>Connector</code> class extended from the <code>Connector</code> Application Block.
	 */
	public AgentServer(String configServerHost,  String   configServerPort,
			String configLayerUserName,	String configLayerPassword,
			String applicationName) {
		
		statusMap = new HashMap();
		listenerVector = new Vector();
		placeVector = new Vector();
			
	
        try{
        	serverData = new ApplicationParameters();
        	connector = new Connector(this);
        	System.out.println("server "+configServerHost+" port "+configServerPort+
        			" user: "+configLayerUserName+" pass "+configLayerPassword+" app "+applicationName);
        	int port = new Integer(configServerPort).intValue();
        	
        	serverData.init(configServerHost,port,null,0,configLayerUserName,configLayerPassword,
        			applicationName, ApplicationParameters.ApplicationType.SERVER,60,3,30 );
			connector.init(serverData);
			
			factory= connector.getAIL();
			System.out.println("connected");
		        	
			connected = "connected";
			buildStatusMap();
			GetPlaceInfoThread p = new GetPlaceInfoThread();
			p.start();
		} catch (RequestFailedException e) {
			e.printStackTrace();
			notifyListeners("Agent Server init error: " + e.getMessage());
		}
		
	}
	/**
	 * Launches a thread that registers for stat events.
	 * @param isServiceAvailable true if the stat service is available.
	 */
	public void setMonitorListener(boolean isServiceAvailable)
	{
		isStatServiceAvailable = isServiceAvailable;
		if(isStatServiceAvailable && monitorListener == null)
		{
			StartMonitorThread p = new StartMonitorThread();
			p.start();
		}
	}
	/**
	 * Returns true if data is available. 
	 * Call the {@link #getAgentStatusMap(int)} or {@link #getPlaceVector()}
	 * @return
	 */
	public boolean isDataAvailable()
	{
		return dataAvailable;
	}
	
	/**
	 * Stops Agent Server.
	 *
	 */
	public void stopAgentServer()
	{
		try {
			if(this.isStatServiceAvailable)
			{
				Iterator agents = statusMap.keySet().iterator();	
				Monitor monitorManager = factory.getMonitor();
				while(agents.hasNext())
				{
					try
					{
						String agent = (String) agents.next();
						monitorManager.unsubscribeStatus(agent, IdObject.ObjectType.PERSON, monitorListener);
					}catch(RequestFailedException __e)
					{
						System.out.println(__e.getMessage());
					}
				
				}
				connector.release();
				}
		} catch (RequestFailedException e) {
			e.printStackTrace();
			this.notifyListeners(e.getMessage());
		}
	}
	/**
	 * Returns the agant status map.
	 */
	public synchronized HashMap getAgentStatusMap(int index)
	{		
		return statusMap;
	}

    private void buildStatusMap()
	{
		System.out.println("Building status map...");
			statusMap= new HashMap();
			Iterator itAgents = factory.getAgentSummaries().iterator();
			while(itAgents.hasNext())
			{
				String id = ((PersonSummary) itAgents.next()).getId();		
				statusMap.put(id, new String[]{"unknown","unknown"});
			}
			System.out.println("Building status map done (size="+statusMap.size()+")");
		
	}
	
	/**
	 * Logs in an agent on a default place.
	 */
	public synchronized String login(String username, String loginId,
            String password, String queue)
	{
		try
		{
			
			Agent m_agent =(Agent)  factory.getPerson (username);
		    if(placeVector.size()==0)
			    return "login action abandonned. No place available";
		
		    String availablePlace = (String) placeVector.get(0);
		    String log_message = new String( );
		
			Place m_place = factory.getPlace(availablePlace);
			m_agent.login(m_place,loginId,
                     password,
                    queue,
                    Dn.Workmode.MANUAL_IN,null,null);
			placeVector.remove(availablePlace);
			
			notifyListeners(availablePlace+" is no longer available.");
			return new String(username+" successful login on"+availablePlace);
		}
		catch(Exception __e)
		{
			return "login failed (error: "+__e.getMessage() +")";
			
		}
		
	}
	
	/**
	 * Logs out an agent.
	 */
	public synchronized String logout(String agent, String queue)
	{
		
		try
		{
			Agent m_agent =(Agent)  factory.getPerson(agent);
			Place m_place = m_agent.getPlace();
			
			String log_message = new String(agent+" multimedia logout: "+m_place.getId()) ;
			m_agent.logout(queue, null, null);
			log_message += " successful";
			placeVector.add(m_place.getId());
			this.notifyListeners(m_place.getId()+ " added to available places");
			return log_message;
		}
		catch(Exception __e)
		{
			return "logout failed (error: "+__e.getMessage() +")";
			
		}		
	}
	
	/** Returns available Places for logging an agent in. */
	public Vector getPlaceVector()
	{
		return this.placeVector;
	}
	/** Adds a listener to the listener list. This listener is associated with a client application. */
	public void addListener(AgentServerListener listener)
	{
		listenerVector.add(listener);
	}
	/** Removes a listener from the listener list. This listener is associated with a client application. */
	
	public void removeListener(AgentServerListener listener)
	{
		listenerVector.remove(listener);
	}
	/**
	 * Sends a message to each listener of the listener list.
	 * @param msg
	 */
	public synchronized void notifyListeners(String msg)
	{
		Iterator itListeners = listenerVector.iterator();
		while(itListeners.hasNext())
		{
			AgentServerListener listener = (AgentServerListener) itListeners.next();
			listener.handleEvent(msg);
		}
	}
	
	/**
	 * Implements the monitor listener. This listener monitors agent statuses. 
	 **/
	class AgentMonitorListener implements MonitorListener
	{
		public void handleMonitorEvent(MonitorEvent event)
		{		
			System.out.println("Event: "+event.getClass().toString());
			
			if(event instanceof MonitorEventAgentStatus)
			{
				
				MonitorEventAgentStatus agentEvent = (MonitorEventAgentStatus) event;				
				NotifyThread th = new NotifyThread(agentEvent);
				th.start();	
				
			}
			
		}
	}
	
	/**
	 * Retrieves a list of default places to be used for logging agents.
	 */
	class GetPlaceInfoThread extends Thread {
	
		public void run()
		{
			Calendar calStart = new GregorianCalendar();			
			System.out.println("Thread for getting default places is started.");
			Iterator itAgents = statusMap.keySet().iterator();	
			
			while(itAgents.hasNext())
			{
				String agentId = (String) itAgents.next();	
				
				Agent myAgent =(Agent)  factory.getPerson (agentId);
				Place p = myAgent.getDefaultPlace() ;
				if(p!= null)
					placeVector.add(p.getId());
				
			}
			dataAvailable = true;
			notifyListeners("Map for available places is ready.");
			
			Calendar calStop = new GregorianCalendar();
			long diff = (calStop.getTimeInMillis() - calStart.getTimeInMillis())/1000;			
			System.out.println("Thread Places terminated ("+ diff+" s.)"+placeVector.toString());
			
			
		}
	
	}
	
	/** Registers listener for the stat service. */
	class StartMonitorThread extends Thread {
		
			public void run()
			{
				Calendar calStart = new GregorianCalendar();			
				
				System.out.println("Thread for adding listeners is started.");
				Monitor monitorManager =null;
				while(monitorManager == null)
				{
					try {
						Thread.sleep(500);
						monitorManager = factory.getMonitor();
						
					} catch (InterruptedException e) {
						e.printStackTrace();
					}
					monitorManager = factory.getMonitor();
				}
				
				Iterator itAgents = factory.getAgentSummaries().iterator();			
				monitorListener = new AgentMonitorListener();
				
				while(itAgents.hasNext())
				{
					PersonSummary itAgentSummary = (PersonSummary) itAgents.next();	
					try
					{
						monitorManager.subscribeStatus(IdObject.ObjectType.PERSON, itAgentSummary.getId(), monitorManager.getChangesBasedNotification(1), monitorListener);
					}
					catch(RequestFailedException __e)
					{
						System.out.println(__e.getMessage());
					}
					
				}
				dataAvailable = true;
				notifyListeners("Agent Server is monitoring agent information.");
				
				Calendar calStop = new GregorianCalendar();
				long diff = (calStop.getTimeInMillis() - calStart.getTimeInMillis())/1000;			
				System.out.println("Thread Listeners terminated ("+ diff+" s.)"+placeVector.toString());
				
				
			}
		
		}
	
	/**
	 * Updates the status map, and notifies all the client applications that are listening to the AgentServer.	
	 */
	class NotifyThread extends Thread {
		MonitorEventAgentStatus agentEvent;
		public NotifyThread(MonitorEventAgentStatus m_agentEvent)
		{
			agentEvent = m_agentEvent;
		}
		public void run()
		{
			MonitorEventAgentStatus.AgentStatus  agentStatus = agentEvent.getStatus();
			HashMap status = getAgentStatusMap(0);
			status.put(agentEvent.getUserName(), new String[]{agentEvent.getPlaceId(), agentStatus.toString()});
			notifyListeners(agentEvent.toString());
		
		}
	}
}
