/*=====================================================================
Any authorized distribution of any copy of this code (including any related documentation) must reproduce the following restrictions, disclaimer and copyright notice:
The Genesys name, trademarks and/or logo(s) of Genesys shall not be used to name (even as a part of another name), endorse and/or promote products derived from this code without prior written permission from Genesys Telecommunications Laboratories, Inc.
The use, copy, and/or distribution of this code is subject to the terms of the Genesys Developer License Agreement.  This code shall not be used, copied, and/or distributed under any other license agreement.
THIS CODE IS PROVIDED BY GENESYS TELECOMMUNICATIONS LABORATORIES, INC. (GENESYS) AS IS WITHOUT ANY WARRANTY OF ANY KIND. GENESYS HEREBY DISCLAIMS ALL EXPRESS, IMPLIED, OR STATUTORY CONDITIONS, REPRESENTATIONS AND WARRANTIES WITH RESPECT TO THIS CODE (OR ANY PART THEREOF), INCLUDING, BUT NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT. GENESYS AND ITS SUPPLIERS SHALL NOT BE LIABLE FOR ANY DAMAGE SUFFERED AS A RESULT OF USING THIS CODE. IN NO EVENT SHALL GENESYS AND ITS SUPPLIERS BE LIABLE FOR ANY DIRECT, INDIRECT, CONSEQUENTIAL, ECONOMIC, INCIDENTAL, OR SPECIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, ANY LOST REVENUES OR PROFITS).
Copyright &copy; 2006 Genesys Telecommunications Laboratories, Inc. All rights reserved.

=====================================================================*/
/***
Agent Interaction Java API 7.5 Code Examples
***/

package agentserver;

import com.genesyslab.ail.AilFactory;
import com.genesyslab.ail.AilLoader;
import com.genesyslab.ail.ServiceStatus;
import com.genesyslab.ail.applicationblocks.connector.ApplicationParameters;
import com.genesyslab.ail.event.ServiceListener;
import com.genesyslab.ail.exception.RequestFailedException;
import com.genesyslab.ail.exception.ServiceException;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;

/**
 * Extends the Connector Application Block modified to handle statistics service.
 * @version 7.5.001.00
 */
public class Connector {

    // use the log4j instead of logger of the AIL
    // but use the same log4j properties than the AIL
    // except if you call AilLoader.noLog4jSettings().
    private static Logger sLog = LogManager.getLogger( "AB.Connector" );

    private AilLoader mAilLoader;
    private ApplicationParameters mAppParam;
    private AilFactory mAilFactory;
    private ServiceStatusListenerAdapter mSTListener;

    private AgentServer agentServer;
    
    /** Creates a new instance of Connector */
    public Connector() {
    }
    
    /** Creates a new instance of Connector and specifies a server to be notified of stat connection status. */
    public Connector(AgentServer server) {
    	agentServer = server;
    	
    }
    
    /**
     * Initializes the connector with application parameters and 
     * connects to the AIL Library.
     * @param applicationParameters the application parameters See {@link ApplicationParameters}.
     * @exception RequestFailedException thrown if problems occured during initialisation.
     */
    public void init(ApplicationParameters applicationParameters)
    throws RequestFailedException {
        if(applicationParameters != null) {
            mAppParam = applicationParameters;
            mAilLoader = new AilLoader(
                    mAppParam.getPrimaryHost(),
                    mAppParam.getPrimaryPort(),
                    mAppParam.getBackupHost(),
                    mAppParam.getBackupPort(),
                    mAppParam.getDefaultUsername(),
                    mAppParam.getDefaultPassword(),
                    mAppParam.getApplicationName(),
                    AilLoader.ApplicationType.getApplicationType(mAppParam.getApplicationType().toInt()),
                    mAppParam.getReconnectionPeriod(),
                    mAppParam.getTimeout());
            mAilLoader.debug();
            mAilLoader.noTrace();
        } else {
            // log into Logger
            sLog.warn("ApplicationParameters is null");
            throw new RequestFailedException("ApplicationParameters is null");
        }
    }
    
    /**
     * Releases the AIL Library instance and all the objects made available through this connector.
     * @exception RequestFailedException thrown if problems occured during release.
     */
    public void release() throws RequestFailedException {
        if(mAilFactory != null) {
            mAilFactory.removeServiceListener(ServiceStatus.Type.AIL, mSTListener);
            mAilFactory.removeServiceListener(ServiceStatus.Type.CHAT, mSTListener);
            mAilFactory.removeServiceListener(ServiceStatus.Type.CONFIG, mSTListener);
            mAilFactory.removeServiceListener(ServiceStatus.Type.DATABASE, mSTListener);
            mAilFactory.removeServiceListener(ServiceStatus.Type.IS, mSTListener);
            mAilFactory.removeServiceListener(ServiceStatus.Type.STAT, mSTListener);
            mAilFactory.removeServiceListener(ServiceStatus.Type.TELEPHONY, mSTListener);
            mAilLoader.killFactory();
            mAilLoader = null;
            mAilFactory = null;
            mSTListener = null;
        } else {
            // log into Logger
            sLog.warn("Connector is not initialised");
            throw new RequestFailedException("Connector is not initialised");
        }
    }
    
    /**
     * Sets the default log file location and default log file name. This location and this file name are
     * for the bootstrap log file only, real location is set in the Configuration Layer, in the options of the application.
     * @param file the name of the log file.
     * @param path the fully-qualified path to the log file ; it can be either absolute or relative.
     * @exception RequestFailedException thrown if problems occured during setting.
     *
     */
    public void setLog(String file, String path) 
    throws RequestFailedException {
        if(mAilLoader != null) {
            if(file != null) {
                mAilLoader.setDefaultLogFileName(file);
            }
            if(path != null) {
                mAilLoader.setDefaultLogFilePath(path);
            }
        } else {
            // log into Logger
            sLog.warn("Connector is not initialised");
            throw new RequestFailedException("Connector is not initialised");
        }
    }

    /**
     * Sets license file when given as application parameter rather than option of config.
     * Must be done before getAIL() which tries to get the licenses.
     * @param license The license file (one or several filepath|port@host separated by semicolon).
     * @throws RequestFailedException if license was already set with a different value.
     */
    public void setLicense(String license)
    throws RequestFailedException {
        if(mAilLoader != null) {
            mAilLoader.setLicenseFile(license);
        } else {
            // log into Logger
            sLog.warn("Connector is not initialised");
            throw new RequestFailedException("Connector is not initialised");
        }
    }
    
    /**
     * Instantiates and returns the <code>AilFactory</code>. If the
     * <code>AIlFactory</code> was not initialized, returns
     * <code>null</code>. In case of error while initializing the
     * connection to the config server, returns <code>null</code>.
     * @return the <code>AilFactory</code> or <code>null</code> in case of error.
     * @exception RequestFailedException thrown if problems occured during setting.
     */
    public AilFactory getAIL() throws RequestFailedException {
        if(mAilLoader != null) {
            mAilFactory = mAilLoader.getAilFactory();
            if(mAilFactory == null) {
                ServiceException _es = AilLoader.getInitException();
                sLog.warn("AilFactory is not initialised");
                throw new RequestFailedException("AilFactory is not initialised " + _es);
            }
            /// Added source code, specific to AgentServer example.
            if(agentServer!=null)
            	mSTListener = new ServiceStatusListenerAdapter(agentServer);
            else
            	mSTListener = new ServiceStatusListenerAdapter();
            mAilFactory.addServiceListener(ServiceStatus.Type.AIL, mSTListener);
            mAilFactory.addServiceListener(ServiceStatus.Type.CHAT, mSTListener);
            mAilFactory.addServiceListener(ServiceStatus.Type.CONFIG, mSTListener);
            mAilFactory.addServiceListener(ServiceStatus.Type.DATABASE, mSTListener);
            mAilFactory.addServiceListener(ServiceStatus.Type.IS, mSTListener);
            mAilFactory.addServiceListener(ServiceStatus.Type.STAT, mSTListener);
            mAilFactory.addServiceListener(ServiceStatus.Type.TELEPHONY, mSTListener);
            return mAilFactory;
        } else {
            // log into Logger
            sLog.warn("Connector is not initialised");
            throw new RequestFailedException("Connector is not initialised");
        }
    }

    /**
     * Implements ServiceListener.
     * This listener implements all the methods that observe the behaviour
     * of the current AIL.
     */
    public class ServiceStatusListenerAdapter implements ServiceListener {

    	AgentServer agentServer;
    	
    	public ServiceStatusListenerAdapter()	{ 	}
    	public ServiceStatusListenerAdapter(AgentServer server)
    	{
    	    agentServer = server;	
    	}
        // ServiceListener
        public void serviceStatusChanged(ServiceStatus.Type service_type,
                    		     String service_name,
				     ServiceStatus.Status service_status) {
            // implements your code here
            System.out.println("serviceStatusChanged: " + service_type
                    + " name service=" + service_name
                    + " status=" + service_status);
            // log into Logger
            sLog.info("serviceStatusChanged: " + service_type
                    + " name service=" + service_name
                    + " status=" + service_status);
            /// Added source code specific to AgentServer example
            if(service_type == ServiceStatus.Type.STAT && agentServer != null)
            {
            	if(service_status == ServiceStatus.Status.ON)
            	    agentServer.setMonitorListener(true);
            	else
            		agentServer.setMonitorListener(false);
            }
         }
    }        
}
