/*=====================================================================
Any authorized distribution of any copy of this code (including any related documentation) must reproduce the following restrictions, disclaimer and copyright notice:
The Genesys name, trademarks and/or logo(s) of Genesys shall not be used to name (even as a part of another name), endorse and/or promote products derived from this code without prior written permission from Genesys Telecommunications Laboratories, Inc.
The use, copy, and/or distribution of this code is subject to the terms of the Genesys Developer License Agreement.  This code shall not be used, copied, and/or distributed under any other license agreement.
THIS CODE IS PROVIDED BY GENESYS TELECOMMUNICATIONS LABORATORIES, INC. (GENESYS) AS IS WITHOUT ANY WARRANTY OF ANY KIND. GENESYS HEREBY DISCLAIMS ALL EXPRESS, IMPLIED, OR STATUTORY CONDITIONS, REPRESENTATIONS AND WARRANTIES WITH RESPECT TO THIS CODE (OR ANY PART THEREOF), INCLUDING, BUT NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT. GENESYS AND ITS SUPPLIERS SHALL NOT BE LIABLE FOR ANY DAMAGE SUFFERED AS A RESULT OF USING THIS CODE. IN NO EVENT SHALL GENESYS AND ITS SUPPLIERS BE LIABLE FOR ANY DIRECT, INDIRECT, CONSEQUENTIAL, ECONOMIC, INCIDENTAL, OR SPECIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, ANY LOST REVENUES OR PROFITS).
Copyright &copy; 20072008 Genesys Telecommunications Laboratories, Inc. All rights reserved.
=====================================================================*/
/***
Agent Interaction SDK 7.5 Java Code Examples
***/
package agent.interaction.samples;

import java.awt.event.ActionEvent;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedList;

import javax.swing.AbstractAction;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JTextArea;
import javax.swing.SwingUtilities;

import com.genesyslab.ail.Interaction;
import com.genesyslab.ail.InteractionMailIn;
import com.genesyslab.ail.InteractionMailOut;
import com.genesyslab.ail.Media;
import com.genesyslab.ail.event.InteractionEvent;
import com.genesyslab.ail.exception.RequestFailedException;
import com.genesyslab.ail.workflow.Queue;


/**
 * This example uses a GUI application to show how to answer and reply an e-mail.
 * <br/><br/>
 *
 * One PlaceListener is registered, which receives InteractionEvents.<br/><br/>
 */
public class SimpleEmailInteraction extends SimplePlace {

	/** Interaction for the inbound email (if any) handled by this example. */
	InteractionMailIn sampleEmailIn;
	/** Interaction for a reply email handled by this example */
	InteractionMailOut sampleEmailReply;


	JButton acceptButton;
	JButton replyButton;
	JButton emailReleaseButton;
	JButton emailDoneButton;
	JButton sendButton;

	JTextArea inboundEmailTextArea;
	JTextArea responseTextArea;
	JFrame replyWindow;

	// main method
	public static void main(String[] args) {
		//Schedule a job for the event-dispatching thread:
		//creating and showing this application's GUI.
		SwingUtilities.invokeLater(new Runnable() {
			public void run() {
				SimpleEmailInteraction application = new SimpleEmailInteraction(
				"Agent Interaction (Java): Simple Email Interaction");
			}
		});
	}
	/**
	 * Creates an instance of SimpleEmailInteraction.<br/>
	 *
	 * The super constructor connects to the Agent Interaction Layer.
	 *
	 * After connecting, it sets up an agent, place, media, and DN for use by
	 * the samples. Then it creates the GUI, links the GUI components (widgets)
	 * to API functionality, and starts the application.
	 *
	 * @param windowTitle
	 *            The title of the GUI window for this example.
	 */
	public SimpleEmailInteraction(String windowTitle) {
		super(windowTitle);
	}
	/**
	 * Tells the AgentInteractionGui class which example has called
	 * the GUI, so that it can determine which widgets will be enabled when the
	 * GUI is created.
	 */
	public void setSampleType() {
		sampleType = "SimpleEmailInteraction";
		
		// Collection of media types for multimedia methods
        mediaList = new LinkedList();
        mediaList.add("email");
        voice = false;
    	
	}
	/**
	 * Links widgets (user interface
	 * components) to API functionality and registers a listener for
	 * events on the Place the agent has logged in to.
	 * <br/><br/>
	 * The linked widgets for Simple Email are buttons for processing
	 * an inbound email and the reply to this email.
	 * These buttons correspond to the following actions:<ul>
	 * <li>Accept</li>
	 * <li>Reply</li>
	 * <li>Reject</Sli>
	 * <li>Mark done</li>
	 * </ul>
	 * @see SimplePlace#linkWidgetsToGui()
	 */
	public void linkWidgetsToGui() {
		super.linkWidgetsToGui();

		// Add an Accept button for getting emailIn
		acceptButton = agentInteractionGui.acceptButton;
		acceptButton.setAction(new AbstractAction("Accept") {
			public void actionPerformed(ActionEvent actionEvent) {
				try {
					sampleEmailIn.answerCall(null);

				} catch (Exception exception) {
					exception.printStackTrace();
				}
			}
		});

		// Add a Reply button
		replyButton = agentInteractionGui.replyButton;
		replyButton.setAction(new AbstractAction("Reply") {
			public void actionPerformed(ActionEvent actionEvent) {
				try {
					// Spawn a reply window
					createReply();
				} catch (Exception exception) {
					exception.printStackTrace();
				}
			}
		});

		// Add an (Email Release) button for the emailIn
		// This feature enables the agent to reject an e-mail
		// so that the email is queued and/or rerouted to another agent
		emailReleaseButton = agentInteractionGui.emailReleaseButton;
		emailReleaseButton.setAction(new AbstractAction("Reject") {
			public void actionPerformed(ActionEvent actionEvent) {
				try {
					sampleEmailIn.releaseCall(null);
				} catch (Exception exception) {
					exception.printStackTrace();
				}
			}
		});

		// Add an (Email Done) button for the emailIn
		emailDoneButton = agentInteractionGui.emailDoneButton;
		emailDoneButton.setAction(new AbstractAction("Done") {
			public void actionPerformed(ActionEvent actionEvent) {
				try {
					sampleEmailIn.markDone();
				} catch (Exception exception) {
					exception.printStackTrace();
				}
			}
		});

		acceptButton.setEnabled(false);
		replyButton.setEnabled(false);
		emailReleaseButton.setEnabled(false);
		emailDoneButton.setEnabled(false);

		inboundEmailTextArea = agentInteractionGui.inboundEmailTextArea;

	}
	
	 public void setPlaceWidgetState()
	    {
	    	if(sampleEmail != null)
	    	{
	    		loginButton.setEnabled(sampleEmail.isPossible(Media.Action.LOGIN));
	    		logoutButton.setEnabled(sampleEmail.isPossible(Media.Action.LOGOUT));
	    		readyButton.setEnabled(sampleEmail.isPossible(Media.Action.READY));
	    		notReadyButton.setEnabled(sampleEmail.isPossible(Media.Action.NOTREADY));
	    		afterCallWorkButton.setEnabled(false);

	    		//Set status to the table.
	    		agentInteractionGui.setPlaceStatus(sampleEmail.getName(),sampleEmail.getStatus().toString());

	    	} 	else {

	    		loginButton.setEnabled(true);
	    		logoutButton.setEnabled(false);
	    		readyButton.setEnabled(false);
	    		notReadyButton.setEnabled(false);
	    		afterCallWorkButton.setEnabled(false);
	    		
	    		agentInteractionGui.setPlaceStatus("email","logged out");
	    		
	    	}
	    }
	/**
	 * Creates a reply interaction (sampleEmailReply) and the associated panel,
	 * so that the user can fill in the email text then send it.
	 */
	public void createReply() {

		try {
			// Getting a queue to which to send the reply email.
			String outboundQueue = getOutboundQueues();

			// Create a reply interaction for the inbound email handled by the example.
			sampleEmailReply = sampleEmailIn.reply(
					outboundQueue, false);

			// Create a dialog box to enable the user to enter the reply text.
			// The subject, the addresses, and part of the message are already created
			// in the interaction.
			agentInteractionGui.createReplyWindow(sampleEmailReply.getSubject(),
					sampleEmailReply.getFromAddress().toString(),
					sampleEmailReply.getMessageText());

			// Linking widgets
			sendButton = agentInteractionGui.sendButton;
			replyWindow = agentInteractionGui.replyWindow;
			responseTextArea = agentInteractionGui.responseTextArea;

			//Setting an action to the Send button.
			sendButton.setAction(new AbstractAction("Send") {
				public void actionPerformed(ActionEvent actionEvent) {
					try {
						// Get the queue to which to send the reply
						String outboundQueue = getOutboundQueues();
						// Set the message text to the reply
						sampleEmailReply.setMessageText(responseTextArea.getText());

						// Send the response
						sampleEmailReply.send(outboundQueue);

						//Close the dialog box
						replyWindow.dispose();
					} catch (Exception exception) {
						exception.printStackTrace();
					}
				}
			});
		} catch (Exception exception) {
			exception.printStackTrace();
		}
	}


	/**
	 * This class receives InteractionEvents, which are handled by the
	 * handleInteractionEvent() method. This class also has widgets whose state
	 * must be set when it receives these events.
	 *
	 * This method enables or disables the specified widgets, which are normally
	 * just buttons and radio buttons, but may include other widgets. To do
	 * this, it uses the isPossible() method of the Possible interface.
	 *
	 * The isPossible() method returns either true or false, depending on
	 * whether a particular action is possible. This boolean value is used to
	 * determine whether the widget will be enabled.
	 */
	public void setInteractionWidgetState() {

		if (sampleEmailIn != null) {
			acceptButton.setEnabled(sampleEmailIn
					.isPossible(InteractionMailIn.Action.ANSWER_CALL));
			replyButton.setEnabled(sampleEmailIn
					.isPossible(InteractionMailIn.Action.REPLY));
			emailReleaseButton.setEnabled(sampleEmailIn
					.isPossible(InteractionMailIn.Action.RELEASE_CALL));
			emailDoneButton.setEnabled(sampleEmailIn
					.isPossible(InteractionMailIn.Action.MARK_DONE));

		} else
		{
			acceptButton.setEnabled(false);
			replyButton.setEnabled(false);
			emailReleaseButton.setEnabled(false);
			emailDoneButton.setEnabled(false);
			inboundEmailTextArea.setText("To accept an incoming e-mail, click the Accept button.");
		}
		if(sampleEmailReply != null)
		{
			sendButton.setEnabled(sampleEmailReply.isPossible(InteractionMailOut.Action.SEND));
		}
	}

	/**
	 *
	 * Receives InteractionEvents, which report changes to the status or data
	 * for all interactions of this place.
	 * If you implement the PlaceListener interface, you must
	 * implement this method. <br/><br/>
	 *
	 * The SimpleEmailInteraction code example uses this method to log these
	 * InteractionEvents as they occur and to set the state of the widgets
	 * used by the SimpleEmailInteraction example. All processing is performed in a separated threadthe EmailInteractionEventThread class.
	 */
	public void handleInteractionEvent(InteractionEvent event) {
		super.handleInteractionEvent(event);
		EmailInteractionEventThread p = new EmailInteractionEventThread(event);
		p.start();		
	}

	/**
	 * Displays the 'From', 'Subject', and 'Text' fields of an inbound e-mail.
	 *
	 */
	public void displaySampleEmailIn()
	{
		try {
			String emailFromAddress = sampleEmailIn.getFromAddress().toString();
			String emailSubject = sampleEmailIn.getSubject();
			String emailText = "From: "
				+ sampleEmailIn.getFromAddress().toString()
				+ "\nSubject: " + sampleEmailIn.getSubject()
				+ "\n__________________________\n\n"
				+ sampleEmailIn.getMessageText();
			inboundEmailTextArea.setText(emailText);
		} catch (RequestFailedException exception) {
			exception.printStackTrace();
		}
	}
	/**
	 * Return an outbound queue to create and/or send a reply e-mail interaction.
	 * @return
	 */
	public String getOutboundQueues() {
		// This method shows how to find the available queues for sending a
		// reply

		String queueId = "";
		Collection availableQueues = sampleEmailIn
		.getAvailableQueuesForChildInteraction();
		Iterator iterator = availableQueues.iterator();
		Queue queue;
		while (iterator.hasNext()) {
			queue = (Queue) iterator.next();
			queueId = queue.getId();
			break;
		}
		return queueId;
	}
	
	
	class EmailInteractionEventThread extends Thread
	{
		InteractionEvent event;
		
		public EmailInteractionEventThread(InteractionEvent _event)
		{
			event=_event;
			
		}
		
		public void run()
		{
			try {
				// 1. test whether or not this is a new interaction
				// to process.
				// If sampleEmailIn is null, the event may notify
				// a new inbound email.
				if (sampleEmailIn == null
						&& event.getSource() instanceof InteractionMailIn
						&& event.getStatus() == Interaction.Status.RINGING) {

					sampleEmailIn = (InteractionMailIn) event.getSource();

					//Displaying From and Subject fields
					String emailText = "From: "
						+ sampleEmailIn.getFromAddress().toString()
						+ "\nSubject: " + sampleEmailIn.getSubject();

					inboundEmailTextArea.setText(emailText);

				}

				//2. The event involves the inbound email being processed
				if(sampleEmailIn!= null && event.getSource().getId() == sampleEmailIn.getId())
				{
					//	The inbound email is in talking status, and can be displayed
					if (event.getStatus() == Interaction.Status.TALKING)
					{
						sampleEmailIn = (InteractionMailIn) event.getSource();
						displaySampleEmailIn();
					}
					// The interaction is processed, the example no longer needs to handle it
					else if (event.getStatus() == Interaction.Status.IDLE )
					{
						if(!sampleEmailIn.isDone())
							sampleEmailIn.markDone();
						sampleEmailIn = null;
					}
					setInteractionWidgetState();
				}
				//3. The event involves the reply email
				else if(sampleEmailReply!= null && event.getSource().getId() == sampleEmailReply.getId())
				{
					if (event.getStatus() == Interaction.Status.IDLE )
					{
						if(!sampleEmailReply.isDone())
							sampleEmailReply.markDone();
						sampleEmailReply = null;
						
						sendButton = null;
						replyWindow = null;
					}
					setInteractionWidgetState();
				}
			} catch (RequestFailedException exception) {
				exception.printStackTrace();
			}

		}
		
	}
}