/*=====================================================================
Any authorized distribution of any copy of this code (including any related documentation) must reproduce the following restrictions, disclaimer and copyright notice:
The Genesys name, trademarks and/or logo(s) of Genesys shall not be used to name (even as a part of another name), endorse and/or promote products derived from this code without prior written permission from Genesys Telecommunications Laboratories, Inc.
The use, copy, and/or distribution of this code is subject to the terms of the Genesys Developer License Agreement.  This code shall not be used, copied, and/or distributed under any other license agreement.
THIS CODE IS PROVIDED BY GENESYS TELECOMMUNICATIONS LABORATORIES, INC. (GENESYS) AS IS WITHOUT ANY WARRANTY OF ANY KIND. GENESYS HEREBY DISCLAIMS ALL EXPRESS, IMPLIED, OR STATUTORY CONDITIONS, REPRESENTATIONS AND WARRANTIES WITH RESPECT TO THIS CODE (OR ANY PART THEREOF), INCLUDING, BUT NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT. GENESYS AND ITS SUPPLIERS SHALL NOT BE LIABLE FOR ANY DAMAGE SUFFERED AS A RESULT OF USING THIS CODE. IN NO EVENT SHALL GENESYS AND ITS SUPPLIERS BE LIABLE FOR ANY DIRECT, INDIRECT, CONSEQUENTIAL, ECONOMIC, INCIDENTAL, OR SPECIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, ANY LOST REVENUES OR PROFITS).
Copyright &copy; 2007-2008 Genesys Telecommunications Laboratories, Inc. All rights reserved.
=====================================================================*/
package agent.interaction.samples.outbound;

import java.awt.event.ActionEvent;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;

import javax.swing.AbstractAction;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JTextArea;

import org.apache.log4j.Logger;


import com.genesyslab.ail.Dn;
import com.genesyslab.ail.OutboundCampaignInfo;
import com.genesyslab.ail.OutboundService;
import com.genesyslab.ail.Place;
import com.genesyslab.ail.event.CampaignEvent;

/**
 * Handles campaign actions and events for the place of the example.<br>
 * In this example, the gui shows the list of campaigns active on the place.
 * The user can select of them to see its status and description. 
 * 
 */
public class CampaignAdapter  {

	/** Adds logs labeled "ForOutbound.CampaignAdapter" in the log file */ 
    Logger sLog = Logger.getLogger("ForOutbound.CampaignAdapter");

	// Agent, place, and DN for the AIL examples
	/** Place object used in this example. */
	public Place samplePlace;
	/** Dn object used in this example. */
	public Dn sampleDn;
	
	OutboundService outboundService;
	AgentOutboundGui outboundGui ;

	JComboBox campaignComboBox;
	JLabel campaignModeLabel;
	JLabel campaignStatusLabel;
	JTextArea campaignDescriptionTextArea;
	JButton requestPreviewRecordButton;
	JButton startPreviewButton;
	JButton stopPreviewButton;

	String selectedCampaignName;
	HashMap<String, String> campaignIDs = new HashMap<String, String>();
    /**
     * Constructor. At creation, the instance links widgets (campaign buttons and box) to campaign actions.
     *
     */
	public CampaignAdapter(Place _samplePlace,Dn _sampleDn, OutboundService _outboundService, AgentOutboundGui _gui) {

		// Set up an agent, place and DN for use by the examples
		samplePlace = _samplePlace;
		sampleDn = _sampleDn;
		outboundGui = _gui;
		outboundService =_outboundService;

		// Link the GUI components to API functionality
		linkWidgetsToGui();

	}
	/**
	 * Adds a campaign to the map of displayed campaigns.
	 */
	private String addCampaignID(String campaignName, String campaignID)
	{		
		if(campaignIDs.containsKey(campaignName))
		{
			if (campaignIDs.containsValue(campaignID))
				return campaignName;
			else
				return addCampaignID(campaignName+"(i)", campaignID);
		}else
		{
			campaignIDs.put(campaignName, campaignID);
			return campaignName;
		}
	
	}
	/**
	 * Removes a campaign from the map of displayed campaigns.
	 */
	private String removeCampaignID(String campaignName, String campaignID)
	{
		
		if( campaignIDs.containsValue(campaignID))
		{
			Iterator names = campaignIDs.keySet().iterator();
			Iterator ids = campaignIDs.values().iterator();
			
			while(names.hasNext() && ids.hasNext())
			{
				String name = (String) names.next();
				String id = (String) ids.next();
				
				if( id.equals(campaignID))
				{
					campaignIDs.remove(name);
					return name;
				}
			}			
		}
		return null;
	}
	
	/**
	 * Links campaign buttons to <code>OutboundService</code> and <code>OutboundCampaignInfo</code> methods.<br>
	 */	
	private void linkWidgetsToGui() {
		
		OutboundCampaignInfo info_cpg = null;
		campaignComboBox = outboundGui.campaignComboBox;
		 try {
         	Collection infos = outboundService.getCampaignInfos();
            for (Iterator iterator = infos.iterator(); iterator.hasNext();) {
            	  info_cpg = ( OutboundCampaignInfo) iterator.next();
            	  String name = addCampaignID(info_cpg.getCampaignName(),info_cpg.getId());
            	  campaignComboBox.addItem(name);
            	  
            	  sLog.info("Filled Campaign Combo box");
            	 
            }

         } catch (Exception exception) {
             exception.printStackTrace();
             sLog.error(exception.getMessage()
             		+ "cause: " + exception.getCause());
         }

         if(info_cpg != null)
         {      	 
        	campaignComboBox.setSelectedItem(0);
        	selectedCampaignName= (String) campaignComboBox.getSelectedItem();
         }
        	 
		campaignModeLabel= outboundGui.campaignModeLabel;
		campaignStatusLabel= outboundGui.campaignStatusLabel;
		campaignDescriptionTextArea= outboundGui.campaignDescriptionTextArea;
		campaignDescriptionTextArea.setEditable(false);
		
		requestPreviewRecordButton= outboundGui.requestPreviewRecordButton;
		requestPreviewRecordButton.setEnabled(false);
		requestPreviewRecordButton.setAction(new AbstractAction("Request preview") {
	            public void actionPerformed(ActionEvent actionEvent) {
	                try {
	                	String ID = campaignIDs.get(selectedCampaignName);
	                	OutboundCampaignInfo info = outboundService.getCampaignInfo(ID);
	                	info.getPreviewRecord();	                	

	                } catch (Exception exception) {
	                	sLog.error(exception.getMessage()
	                     		+ "cause: " + exception.getCause());
	                }
	            }
	        });

		startPreviewButton= outboundGui.startPreviewButton;
		startPreviewButton.setEnabled(false);
		startPreviewButton.setAction(new AbstractAction("Start preview") {
            public void actionPerformed(ActionEvent actionEvent) {
                try {
                	String ID = campaignIDs.get(selectedCampaignName);
                	OutboundCampaignInfo info = outboundService.getCampaignInfo(ID);
                	info.startPreviewMode();	 	                	

                } catch (Exception exception) {
                	sLog.error(exception.getMessage()
                     		+ "cause: " + exception.getCause());
                }
            }
        });

		
		stopPreviewButton= outboundGui.stopPreviewButton;
		stopPreviewButton.setEnabled(false);
		stopPreviewButton.setAction(new AbstractAction("Stop preview") {
            public void actionPerformed(ActionEvent actionEvent) {
                try {
                	String ID = campaignIDs.get(selectedCampaignName);
                	OutboundCampaignInfo info = outboundService.getCampaignInfo(ID);
                	info.stopPreviewMode();	                	

                } catch (Exception exception) {
                	sLog.error(exception.getMessage()
                     		+ "cause: " + exception.getCause());
                }
            }
        });

		setCampaignWidgetState();

		campaignComboBox.addItemListener(new CampaignSelectionListener());
	}

	/**
	 * Enables/Disables widgets according to the data of the selected campaign.
	 */
	public synchronized void setCampaignWidgetState()
	{
		if(selectedCampaignName == null)
		{
			if(campaignIDs.size()>0)
			{
				campaignComboBox.setSelectedItem(0);
				selectedCampaignName= (String) campaignComboBox.getSelectedItem();
			}
			
		}
		if(selectedCampaignName != null)
		{
			try {
				String ID = campaignIDs.get(selectedCampaignName);
            	OutboundCampaignInfo info = outboundService.getCampaignInfo(ID);
            	
            	campaignModeLabel.setText("Mode: "+info.getCampaignMode().toString());
            	campaignDescriptionTextArea.setText("Status: "+info.getCampaignState().toString() + "\n"+info.getCampaignDescription());
            	startPreviewButton.setEnabled(info.isPossible(OutboundCampaignInfo.Action.START_PREVIEW_MODE));
            	stopPreviewButton.setEnabled(info.isPossible(OutboundCampaignInfo.Action.STOP_PREVIEW_MODE));
            	requestPreviewRecordButton.setEnabled(info.isPossible(OutboundCampaignInfo.Action.REQUEST_PREVIEW_RECORD));
			
			
			} catch (Exception exception) {
				sLog.error(exception.getMessage()
	             		+ "cause: " + exception.getCause());
			}
		}else
		{
			campaignModeLabel.setText("Mode: ");
    		campaignDescriptionTextArea.setText("");
		
        	startPreviewButton.setEnabled(false);
        	stopPreviewButton.setEnabled(false);
        	requestPreviewRecordButton.setEnabled(false);
		}

	}
	/**
	 * Receives CampaignEvents, which report changes to the data of a campaign. Because this method is called in the PlaceListener, 
	 * it creates a thread to manage the event without delaying the publication of further incoming events. 
	 */
	public void handleCampaignEvent(CampaignEvent event) {

		CampaignEventThread p = new CampaignEventThread(event);
		p.start();

	}

	/**
	 * Manages one CampaignEvent by updating boxes and widgets states.
	 *
	 */		
	class CampaignEventThread extends Thread {

		CampaignEvent event;

		public CampaignEventThread(CampaignEvent _event) {
			event= _event;
		}


		public void run()
		{
			OutboundCampaignInfo info = event.getCampaignInfo();
			sLog.debug("CampaignEventThread "+ info.toString());

			switch (event.getEventType().toInt())
			{
				case CampaignEvent.Type.CAMPAIGN_ADDED_:
					if(!campaignIDs.containsValue(info.getId()))
					{
						String campaignName =addCampaignID(info.getCampaignName(), info.getId());
						campaignComboBox.addItem(campaignName);
					}				
					break;
			case CampaignEvent.Type.CAMPAIGN_REMOVED_:
				if(campaignIDs.containsValue(info.getId()))
				{
					String removed = removeCampaignID(info.getCampaignName(), info.getId());
					campaignComboBox.removeItem(removed);
					if(selectedCampaignName != null && selectedCampaignName.equals(removed))
					{
						selectedCampaignName = null;
					}					
				}
				break;
			case CampaignEvent.Type.CAMPAIGN_MODE_CHANGED_:
				if(!campaignIDs.containsValue(info.getId()))
				{
					String campaignName =addCampaignID(info.getCampaignName(), info.getId());
					campaignComboBox.addItem(campaignName);
				}
				break;
			case CampaignEvent.Type.CAMPAIGN_STATE_CHANGED_:
				if(!campaignIDs.containsValue(info.getId()))
				{
					String campaignName =addCampaignID(info.getCampaignName(), info.getId());
					campaignComboBox.addItem(campaignName);
				}
				break;			
			}
			setCampaignWidgetState();
		}
	}

	/**
	 * Listens selection events in the campaign list combobox. When the user selects a campaign name in this combobox, the campaign panel is updated accordingly.
	 */
	class CampaignSelectionListener implements ItemListener {
			public void itemStateChanged(ItemEvent e) {
				JComboBox cb = (JComboBox)e.getSource();
				selectedCampaignName = (String)cb.getSelectedItem();
				setCampaignWidgetState();
				
			}


	}

}
