/*=====================================================================
Any authorized distribution of any copy of this code (including any related documentation) must reproduce the following restrictions, disclaimer and copyright notice:
The Genesys name, trademarks and/or logo(s) of Genesys shall not be used to name (even as a part of another name), endorse and/or promote products derived from this code without prior written permission from Genesys Telecommunications Laboratories, Inc.
The use, copy, and/or distribution of this code is subject to the terms of the Genesys Developer License Agreement.  This code shall not be used, copied, and/or distributed under any other license agreement.
THIS CODE IS PROVIDED BY GENESYS TELECOMMUNICATIONS LABORATORIES, INC. (GENESYS) AS IS WITHOUT ANY WARRANTY OF ANY KIND. GENESYS HEREBY DISCLAIMS ALL EXPRESS, IMPLIED, OR STATUTORY CONDITIONS, REPRESENTATIONS AND WARRANTIES WITH RESPECT TO THIS CODE (OR ANY PART THEREOF), INCLUDING, BUT NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT. GENESYS AND ITS SUPPLIERS SHALL NOT BE LIABLE FOR ANY DAMAGE SUFFERED AS A RESULT OF USING THIS CODE. IN NO EVENT SHALL GENESYS AND ITS SUPPLIERS BE LIABLE FOR ANY DIRECT, INDIRECT, CONSEQUENTIAL, ECONOMIC, INCIDENTAL, OR SPECIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, ANY LOST REVENUES OR PROFITS).
Copyright &copy; 2007-2008 Genesys Telecommunications Laboratories, Inc. All rights reserved.
=====================================================================*/

package media.sdk.java.examples;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.GregorianCalendar;

import com.genesyslab.omsdk.mil.MILFactory;
import com.genesyslab.omsdk.mil.MILISInteractionParameters;
import com.genesyslab.omsdk.mil.MILInitializationParameters;
import com.genesyslab.omsdk.mil.MILInteraction;
import com.genesyslab.omsdk.mil.MILInteractionManager;
import com.genesyslab.omsdk.mil.MILUCSInteractionParameters;
import com.genesyslab.omsdk.mil.MILUCSInteractionStatus;
import com.genesyslab.omsdk.mil.MILUCSManager;
import com.genesyslab.omsdk.mil.exception.MILInitializationException;

/**
 * Implements a simple server application that reads the content of a source directory.
 * When parsing the directory, it creates a thread for processing each file.
 * Each thread creates a MILInteraction, adds the file as attached data to the interaction,
 * submits the interaction, and deletes the file from the source directory.
 */
public class SimpleMediaServer extends SimpleConnector implements Runnable {
	private File sourcePath;
	private File savePath;
	private File[] filesToSend;
	/** The queue name to which media interactions are submitted. */
	public String queueName;
	/** The media type for interactions to be submitted. */
	public String mediaType;
	/** The interaction type for interactions to be submitted. */
	public String interactionType;
	/** The interaction subtype for interactions to be submitted. */
	public String interactionSubtype;


	MILInteractionManager ixnManager;
	MILUCSManager ucsManager;

	Calendar calendar = new GregorianCalendar();
    DateFormat timestamp = new SimpleDateFormat("yyyy-dd-MM HH:mm:ss:SSS");

    /**
     * Constructor.
     * @param queue_name queue name to which media interactions are submitted.
     * @param media_type media type possible values are defined in the Configuration Layer as "Media Type" attribute values.
     * @param interaction_type interaction type; possible values are defined in the Configuration Layer as "Interaction Type" attribute values.
     * @param interaction_subtype interaction subtype; possible values are defined in the Configuration Layer as "Interaction Type" attribute values.
     * @param source_path source directory containing input files to attach to the created interactions.
     * @param save_path directory in which to save files for which interaction creation failed.
     */
	public SimpleMediaServer(
			 String queue_name,
			 String media_type,
			 String interaction_type,
			 String interaction_subtype,
			 String source_path,
			 String save_path,
			 boolean custom) {
		super();

		queueName = queue_name;
		mediaType = media_type;
		interactionType = interaction_type;
		interactionSubtype = interaction_subtype;
		sourcePath = new File(source_path);
		savePath = new File(save_path);


		try {

			MILInitializationParameters milParam;
			/// If you are running StartCustomMedia, custom is true
			if( custom == true)
				milParam = new MILInitializationParameters(new String[]{"media.sdk.java.examples.SimpleCustomExtension"});
			else
				milParam = new MILInitializationParameters(null);

			MILFactory.initialize(milParam);

			//Getting Managers
			ixnManager = MILFactory.getMILFactory().getInteractionManager();
			ucsManager = MILFactory.getMILFactory().getUCSManager();

			savePath.mkdir();

		} catch (MILInitializationException e) {
			e.printStackTrace();
			stop();
		}

	}
	/**
	 * Synchronized method that manages interaction IDs.
	 */
	public synchronized long incrementID()
	{
	         return System.currentTimeMillis();

	}
	/**
	 * This method periodically reads the source directory and process the file list.
	 */
	public void run()
	{
		int i= 0;
		while(i<1)
		{
			getSourceFiles();
			processFileList();
			try {
				Thread.sleep(2000);
			} catch (InterruptedException e) {
				e.printStackTrace();
				stop();

			}
		}


	}
	/**
	 * Reads the content of the <code>sourcePath</code> directory.
	 *
	 */
	public synchronized void getSourceFiles()
	{
		filesToSend = sourcePath.listFiles();

	}
	/**
	 * Creates
	 * a thread for processing each file found in the filesToSend list.
	 *
	 */
	public synchronized void processFileList()
	{
		for(int i=0; i<filesToSend.length; i++)
		{
			//creates a thread for each file to send
			SubmitThread p = new SubmitThread(filesToSend[i]);
			p.start();
		}

	}
	/**
	 * Stops the server and properly release factories.
	 */
	public void stop()
	{
		try{

			System.out.println(createTimeStamp()+" Exiting...");
			MILFactory.releaseFactory();
			super.stop();
			System.exit(0);

		}catch(Exception __e)
		{
			__e.printStackTrace();
		}
	}
	/**
	 * Creates and submits an interaction
	 * containing the fileToSend file as an attached data.
	 */
	class SubmitThread extends Thread {

		File fileToSend;
		SubmitThread(File _fileToSend) {
            this.fileToSend = _fileToSend;
        }


        public void run() {

        	//Reading the file
        	FileInputStream fileInputStream;
        	try {
        		//Reading the file
        		fileInputStream = new FileInputStream(fileToSend);
        		byte[] b = new byte[51200];
        		System.out.println(createTimeStamp()+ " Processing "+fileToSend.getName()+" - Read: "+fileInputStream.read(b));
        		fileInputStream.close();

        		// Saving the file
        		// If the interaction submission fails,
    			// the saved file is not deleted
        		File saveFile = File.createTempFile(fileToSend.getName(), ".sav",savePath);
				FileOutputStream fileOutputStream = new FileOutputStream(saveFile);
				fileOutputStream.write(b);
				fileOutputStream.close();

				//Deleting the original file, so that it is submitted once only
				if(! fileToSend.delete())
					System.out.println(createTimeStamp()+fileToSend.getName()+" Deletion aborted.");

        		//Processing the file
        		try {

        			//Creating the interaction
        			MILInteraction myInteraction =
        				ixnManager.createInteraction(interactionType,interactionSubtype,mediaType);

        			//Setting Interaction properties
        			myInteraction.setReceivedAt(new GregorianCalendar().getTime());
        			//String ID = IDRoot+incrementID();
        			String ID = ""+incrementID();
        			myInteraction.setID(ID);
        			myInteraction.setQueueName(queueName);

        			//******************** for UCS ********************************
        			//Comment this section if you do not have a running UCS
        			//Attaching the file to ucs user data
        			MILUCSInteractionParameters ixnUCSparam = myInteraction.getUCSParameters();
        			ixnUCSparam.setProperty("FileName", fileToSend.getName());
        			ixnUCSparam.setProperty("FileBody", b);
        			ixnUCSparam.setProperty("Subject", "MIL interaction from Simple Media Server");
        			ixnUCSparam.setStatus(MILUCSInteractionStatus.IN_PROCESS);

        			System.out.println(createTimeStamp()+"Save: "+myInteraction.getID() + " "
        					+ myInteraction.getMediaType()+" "
							+ myInteraction.getQueueName()+" "
							+ myInteraction.getReceivedAt().toGMTString());

        			//Saving the interaction in UCS
        			ucsManager.saveInteraction(myInteraction);

        			//********************* for IS **********************************
        			MILISInteractionParameters ixnISParam = myInteraction.getISParameters();

        			//Attaching the file to is user data
        			ixnISParam.setProperty("FileName", fileToSend.getName());
        			ixnISParam.setProperty("FileBody", b);
        			ixnISParam.setProperty("Subject", "MIL interaction from Simple Media Server");

        			// Printing object content
        			System.out.println(createTimeStamp()+ "Submit: "+myInteraction.getID() + " "
        					+ myInteraction.getMediaType()+" "
							+ myInteraction.getQueueName()+" "
							+ myInteraction.getReceivedAt().toGMTString());

        			// Submitting the interaction
        			ixnManager.submit(myInteraction);

        			//Deleting the saved file
        			if(! saveFile.delete())
        			{
        				System.out.println(createTimeStamp()+saveFile.getName()+" Deletion aborted.");
        			}

        		} catch (Exception e) {
        			e.printStackTrace();

        		}
        	} catch (Exception e1) {
        		e1.printStackTrace();
        	}
        }
	}

	/**
	 * Returns a timestamp for log messages.
	 */
    protected String createTimeStamp() {
        // Create a time stamp for application log window messages
        return timestamp.format(new GregorianCalendar().getTime())+ " SimpleMediaServer - ";
    }


}
