/*=====================================================================
 Any authorized distribution of any copy of this code (including any related documentation) must reproduce the following restrictions, disclaimer and copyright notice:
 The Genesys name, trademarks and/or logo(s) of Genesys shall not be used to name (even as a part of another name), endorse and/or promote products derived from this code without prior written permission from Genesys Telecommunications Laboratories, Inc.
 The use, copy, and/or distribution of this code is subject to the terms of the Genesys Developer License Agreement.  This code shall not be used, copied, and/or distributed under any other license agreement.
 THIS CODE IS PROVIDED BY GENESYS TELECOMMUNICATIONS LABORATORIES, INC. (GENESYS) AS IS WITHOUT ANY WARRANTY OF ANY KIND. GENESYS HEREBY DISCLAIMS ALL EXPRESS, IMPLIED, OR STATUTORY CONDITIONS, REPRESENTATIONS AND WARRANTIES WITH RESPECT TO THIS CODE (OR ANY PART THEREOF), INCLUDING, BUT NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT. GENESYS AND ITS SUPPLIERS SHALL NOT BE LIABLE FOR ANY DAMAGE SUFFERED AS A RESULT OF USING THIS CODE. IN NO EVENT SHALL GENESYS AND ITS SUPPLIERS BE LIABLE FOR ANY DIRECT, INDIRECT, CONSEQUENTIAL, ECONOMIC, INCIDENTAL, OR SPECIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, ANY LOST REVENUES OR PROFITS).
 Copyright &copy; 2006 Genesys Telecommunications Laboratories, Inc. All rights reserved.
 =====================================================================*/
/**
 Queued Interaction Java API 7.5 Code Examples
 */
package queued.sdk.java.examples;

import java.util.Collection;
import java.util.Iterator;

import javax.swing.SwingUtilities;

import com.genesyslab.omsdk.qil.QILInteraction;
import com.genesyslab.omsdk.qil.QILQueue;
import com.genesyslab.omsdk.qil.QILQueueMonitorStatus;
import com.genesyslab.omsdk.qil.event.QILInteractionEvent;
import com.genesyslab.omsdk.qil.event.QILInteractionListener;
import com.genesyslab.omsdk.qil.event.QILInteractionPropertiesChangedEvent;
import com.genesyslab.omsdk.qil.event.QILInteractionStatusChangedEvent;
import com.genesyslab.omsdk.qil.event.QILQueueContentChangedEvent;
import com.genesyslab.omsdk.qil.event.QILQueueEvent;
import com.genesyslab.omsdk.qil.event.QILQueueMonitorStatusEvent;

/**
 * Monitors interactions.
 * <p>This example activates the interaction list component and displays real-time information for interactions.
 * </p>
 * <p>This example uses the following application blocks:
 * <ul> <li> com.genesyslab.omsdk.qil.applicationblocks.monitoring.listeners.MyQILInteractionListener</li>
 * </ul>
 * @version 7.5.001.01
 */
public class SimpleMonitorInteraction extends SimpleMonitorQueue implements QILInteractionListener {
	
	/**
	 * Creates a <code>SimpleMonitorInteraction</code> instance.
	 * @param windowTitle
	 */
	public SimpleMonitorInteraction(String windowTitle) {
		super(windowTitle);
		
	}
	
	/**
	 * Indicates which example is running.
	 * This method tells the <code>OpenMediaSdkGui</code> class which example has called
	 * the GUI, so that <code>OpenMediaSdkGui</code> can determine which widgets will be enabled when the
	 * GUI is created.
	 */
	public void setExampleType() {
		exampleType = "SimpleMonitorInteraction";
	}
	/**
	 * Updates the widgets associated with the selected queue (if needed).
	 */
	public void setQueueWidgets(String queueName)
	{
		super.setQueueWidgets(queueName);
		//Updating the Queues table
		if(queueName.equals(selectedQueueName)){
			this.sdkGui.switchIxnTable(queueName);			
		}
	}
	/**
	 * Updates the interaction widgets associated with a queue.
	 * @param queueName the queue.
	 * @param added the interactions to be added to the table.
	 * @param removed the interaction to be removed from the table.
	 */
	public void setIxnWidgets(String queueName, Collection added, Collection removed){
		
		Iterator addedIxns = added.iterator();
		//Getting Added Interactions
		while (addedIxns.hasNext())
		{
			QILInteraction ixn = (QILInteraction) addedIxns.next();
			this.sdkGui.setIxn(queueName,ixn.getID(), ixn.getMediaType().toString(), ixn.getType().toString(), ixn.getSubtype().toString(), ixn.getStatus().toString(), ixn.getProperties());
		}
		
		//Getting Removed Interactions
		Iterator deletedIxns = removed.iterator();
		while (deletedIxns.hasNext())
		{
			QILInteraction ixn = (QILInteraction) deletedIxns.next();
			this.sdkGui.removeIxn(queueName, ixn.getID());
		}
		
	}
	
	
	/**
	 * Manages interaction events to update the interaction table and display the event information in the log panel.
	 * <br>This method uses an InteractionEventThread which conforms to the <code>com.genesyslab.omsdk.qil.applicationblocks.monitoring.listeners.MyQILInteractionListener</code> application block.	 
	 * @see com.genesyslab.omsdk.qil.event.QILInteractionListener#handleInteractionEvent(com.genesyslab.omsdk.qil.event.QILInteractionEvent)
	 */
	public void handleInteractionEvent(QILInteractionEvent event) {
		InteractionEventThread p = new InteractionEventThread(event);
		p.start();
		
	}
	
	class InteractionEventThread extends Thread
	{
		QILInteractionEvent event;
		
		public InteractionEventThread(QILInteractionEvent _event)
		{
			event=_event;			
		}
		/////////////// Source from the MyQILInteractionListener Application Block /////////////
		public void run()
		{
			
			QILInteraction interaction = event.getInteraction();
			if (event instanceof QILInteractionStatusChangedEvent) {
				QILInteractionStatusChangedEvent eventStatusChanged = (QILInteractionStatusChangedEvent) event;
				QILInteraction ixn = eventStatusChanged.getInteraction();
				//          Creating a log message
				if(sdkGui.createDetailedIxnEventMessages)
				{
					String detailedLogMessage= "In queue "+ ixn.getQueue().getID() + ": ixn " +ixn.getID()
					+ " ["+ ixn.getMediaType() 
					+ ","+ ixn.getType()
					+ ","+ ixn.getSubtype()
					+ "] ->  New Status ["+eventStatusChanged.getStatus().toString()
					+"]" ;
					sdkGui.writeLogMessage(detailedLogMessage, OpenMediaSdkGui.interactionEventStyle);
					
				}else if (sdkGui.logIxnEvents)
				{
					String logMessage="In queue "+ ixn.getQueue().getID() +": ixn " +ixn.getID()
					+ " ->  New Status ["+eventStatusChanged.getStatus().toString()+"]" ;
					sdkGui.writeLogMessage(logMessage, OpenMediaSdkGui.interactionEventStyle);
				}
				sdkGui.setIxn(ixn.getQueue().getID(), ixn.getID(),ixn.getMediaType().toString(),ixn.getType().toString(),ixn.getSubtype().toString(),eventStatusChanged.getStatus().toString(), ixn.getProperties());
				
			} else if (event instanceof QILInteractionPropertiesChangedEvent) {
				QILInteractionPropertiesChangedEvent eventPropertiesChanged = (QILInteractionPropertiesChangedEvent) event;
				QILInteraction ixn = eventPropertiesChanged.getInteraction(); 
				if(sdkGui.createDetailedIxnEventMessages)
				{
					String detailedLogMessage=  "In queue "+ ixn.getQueue().getID() +": ixn " +ixn.getID() 
					+ " Changed properties "+eventPropertiesChanged.getChangedProperties().toString() 
					+ " Removed properties "+eventPropertiesChanged.getRemovedProperties().toString();
					sdkGui.writeLogMessage(detailedLogMessage, OpenMediaSdkGui.interactionEventStyle);
				}else if (sdkGui.logIxnEvents)
				{
					String logMessage=  "In queue "+ ixn.getQueue().getID() +": ixn " +ixn.getID() 
					+ " Changed properties "+eventPropertiesChanged.getChangedProperties().toString() ;
					sdkGui.writeLogMessage(logMessage, OpenMediaSdkGui.interactionEventStyle);
				}
				sdkGui.setIxn(ixn.getQueue().getID(), ixn.getID(),ixn.getMediaType().toString(),ixn.getType().toString(),ixn.getSubtype().toString(),ixn.getStatus().toString(), ixn.getProperties());
				
			}
		}
	}
	/**
	 * Handles queue events.
	 * This method overloads the parent <code>SimpleMonitorQueue.handleQueueEvent()</code> method to listen the
	 * interaction events of a monitored queue. It also updates the interaction widgets associated with the queue.
	 */
	public void handleQueueEvent(QILQueueEvent event)
	{
		super.handleQueueEvent(event);
		QueueEventForIxnThread p = new QueueEventForIxnThread(event,this);
		p.start();
		
	}
	
	class QueueEventForIxnThread extends Thread
	{
		QILQueueEvent event;
		SimpleMonitorInteraction app;
		
		public QueueEventForIxnThread(QILQueueEvent _event,SimpleMonitorInteraction _app)
		{
			event=_event;	
			app=_app;
		}
		/////////////// Source from the MyQueueListener Application Block /////////////
		public void run()
		{
			if(event instanceof QILQueueMonitorStatusEvent)
			{
				//Getting the event
				QILQueueMonitorStatusEvent statusEvent = (QILQueueMonitorStatusEvent)event;
				QILQueue queue = statusEvent.getQueue();
				//Updating GUI
				if(statusEvent.getNewMonitorStatus() ==  QILQueueMonitorStatus.MONITORED)
				{
					queue.addInteractionListener(app);
				}
				else if(statusEvent.getNewMonitorStatus() ==  QILQueueMonitorStatus.NOT_MONITORED)
				{
					queue.removeInteractionListener(app);
					sdkGui.clearInteractions(queue.getID());
				}
				
			} else if(event instanceof QILQueueContentChangedEvent)
			{
				QILQueueContentChangedEvent changeEvent = (QILQueueContentChangedEvent)event;
				QILQueue queue = changeEvent.getQueue();
				setIxnWidgets(queue.getID(), changeEvent.getAddedInteractions(), changeEvent.getRemovedInteractions());			
			}
		}
	}
	
	
	public static void main(String[] args) {
		//		Schedule a job for the event-dispatching thread:
		//creating and showing this application's GUI.
		SwingUtilities.invokeLater(new Runnable() {
			public void run() {
				SimpleMonitorInteraction application = new SimpleMonitorInteraction(
				"Queued Interaction Java API: Simple Monitor Interaction");
			}
		});
	}
}
