/*=====================================================================
Any authorized distribution of any copy of this code (including any related documentation) must reproduce the following restrictions, disclaimer and copyright notice:
The Genesys name, trademarks and/or logo(s) of Genesys shall not be used to name (even as a part of another name), endorse and/or promote products derived from this code without prior written permission from Genesys Telecommunications Laboratories, Inc.
The use, copy, and/or distribution of this code is subject to the terms of the Genesys Developer License Agreement.  This code shall not be used, copied, and/or distributed under any other license agreement.
THIS CODE IS PROVIDED BY GENESYS TELECOMMUNICATIONS LABORATORIES, INC. (GENESYS) AS IS WITHOUT ANY WARRANTY OF ANY KIND. GENESYS HEREBY DISCLAIMS ALL EXPRESS, IMPLIED, OR STATUTORY CONDITIONS, REPRESENTATIONS AND WARRANTIES WITH RESPECT TO THIS CODE (OR ANY PART THEREOF), INCLUDING, BUT NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT. GENESYS AND ITS SUPPLIERS SHALL NOT BE LIABLE FOR ANY DAMAGE SUFFERED AS A RESULT OF USING THIS CODE. IN NO EVENT SHALL GENESYS AND ITS SUPPLIERS BE LIABLE FOR ANY DIRECT, INDIRECT, CONSEQUENTIAL, ECONOMIC, INCIDENTAL, OR SPECIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, ANY LOST REVENUES OR PROFITS).
Copyright &copy; 2006 Genesys Telecommunications Laboratories, Inc. All rights reserved.
=====================================================================*/
/**
Queued Interaction Java API 7.5 Code Examples
*/
package queued.sdk.java.examples;

import java.awt.event.ActionEvent;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import javax.swing.AbstractAction;
import javax.swing.JButton;
import javax.swing.ListSelectionModel;
import javax.swing.SwingUtilities;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import com.genesyslab.omsdk.qil.QILFactory;
import com.genesyslab.omsdk.qil.QILQueue;
import com.genesyslab.omsdk.qil.QILInteraction;
import com.genesyslab.omsdk.qil.event.QILQueueEvent;
import com.genesyslab.omsdk.qil.event.QILQueueContentChangedEvent;
import com.genesyslab.omsdk.qil.event.QILQueueMonitorStatusEvent;
import com.genesyslab.omsdk.qil.event.QILQueueListener;
import com.genesyslab.omsdk.qil.exception.QILObjectNotFoundException;
import com.genesyslab.omsdk.qil.exception.QILRequestFailedException;
import com.genesyslab.omsdk.qil.exception.QILUninitializedException;

/**
 * Monitors queues.
 * This example uses a GUI application to show how to monitor queues (QILQueue objects):<ul>
 * <li>It displays the annex map for a selected queue in the list of available queues in the
 * Configuration Layer.
 * <li>It enables you to start and stop monitoring an active queue in this list.
 * <li>It implements the QILQueueListener interface to listen for changes in the monitored queues
 * (status and interactions).
 * </ul>
 *
 * <strong>PLEASE NOTE:</strong> This code example will not work unless you
 * edit the configuration data in OpenMediaSdk.properties so that it matches
 * objects that already exist in your local Configuration Layer. <br><br>
 *
 * There are several steps needed to prepare an application to use the services
 * of the Queued Interaction Java Library. This example uses the services of other
 * classes to carry out these functions. <br><br>
 *
 * <ul>
 * <li>{@link OpenMediaSdkData} makes various configuration data available to
 * the Queued Interaction Java Library code examples.</li>
 * <li>{@link OpenMediaSdkGui} creates the common graphical user interface
 * used by the code examples.</li>
 * <li>{@link SimpleConnector} connects code examples to Configuration Layer and Interaction Server.</li>
 * <li>{@link SimpleService} gets and displays the connected services, and the
 * list of queues available, in the Configuration Layer.
 * </li>
 * </ul><br><br>
 * 
 * This code example uses the folowing application blocks:<ul><code>
 * <li>com.genesyslab.omsdk.qil.applicationblocks.monitoring.getters.GetQueue</li>
 * <li>com.genesyslab.omsdk.qil.applicationblocks.monitoring.listeners.StartMonitoringQueue</li>
 * <li>com.genesyslab.omsdk.qil.applicationblocks.monitoring.listeners.MyQueueListener</li></code>
 * </ul>
 * @version 7.5.001.00
 */
public class SimpleMonitorQueue extends SimpleService implements QILQueueListener {

	/** Start button. */
	JButton startButton;
	/** Sop button */
	JButton stopButton;
	/** Selected queue. */
	String selectedQueueName;
	/** Number of interaction per queues */
	HashMap ixnPerQueue ;

	public static void main(String[] args) {
		//Schedule a job for the event-dispatching thread:
		//creating and showing this application's GUI.
		SwingUtilities.invokeLater(new Runnable() {
			public void run() {
				SimpleMonitorQueue application = new SimpleMonitorQueue(
				"Queued Interaction Java API: Simple Monitor Queue");
			}
		});
	}

	/**
	 * Creates an instance of <code>SimpleMonitorQueue</code>.<br>
	 * @param windowTitle
	 *            The title of the GUI window for this example.
	 */
	public SimpleMonitorQueue(String windowTitle) {
		super(windowTitle);
		//Register a ListSelectionListener on the Queues table
		sdkGui.activateQueueRowSelection(new SelectedQueueListener(this));

	}
	/**
	 * Indicates which example is running.
	 * This method tells the OpenMediaSdkGui class which example has called
	 * the GUI, so that OpenMediaSdkGui can determine which widgets will be enabled when the
	 * GUI is created.
	 */
	public void setExampleType() {
		exampleType = "SimpleMonitorQueue";
	}

	/**
	 * Links QIL information to GUI components.
	 * The OpenMediaSdkGui class creates a single graphical user interface
	 * that is used by each of the Queued SDK Java Library code examples.
	 * This GUI contains all of the widgets used by the examples, but they are
	 * only placeholders. That is, they have no functionality when they
	 * are created in the GUI class. <br><br>
	 *
	 * When an example is run, the linkWidgetsToGui() method is used to
	 * associate these widgets with the appropriate fields in the example class,
	 * as shown here: <br><br>
	 *
	 * <code>startButton = sdkGui.startButton;</code> <br><br>
	 *
	 * In many cases, particularly for buttons and radio buttons, these fields
	 * are then linked to actions that correspond to the functions they are
	 * designed to carry out.
	 */
	public void linkWidgetsToGui() {
		super.linkWidgetsToGui();
		ixnPerQueue= new HashMap();
		/* Source from the GetQueue Application Block, refer to the getAllQueues() method */
		try{
			Collection queues = QILFactory.getQILFactory().getAllQueues();
            queues.iterator();
            for (Iterator iterator = queues.iterator(); iterator.hasNext();) {
                QILQueue queue = (QILQueue) iterator.next();
                sdkGui.setQueue(queue.getID(), queue.isMonitored(),new Integer(0).toString());
                ixnPerQueue.put(queue.getID(), new Integer(0));
                queue.addQueueListener(this);
            }
		}		
		catch (QILRequestFailedException __e)
		{
			sdkGui.writeLogMessage("Request failed. Connection to Configuration Server may be lost. "+__e.toString(), OpenMediaSdkGui.errorStyle);
		}	
		
		/* Adding a start button for monitoring */
		startButton = sdkGui.startButton;
		startButton.setAction(new AbstractAction("Start") {
			public void actionPerformed(ActionEvent actionEvent) {
				if(selectedQueueName != null)
				{
					startMonitoring(selectedQueueName);           					
				}		
			}
		});
		startButton.setEnabled(false);
		
		/* Adding a stop button for monitoring */
		stopButton = sdkGui.stopButton;
		stopButton.setAction(new AbstractAction("Stop") {
			public void actionPerformed(ActionEvent actionEvent) {
				if(selectedQueueName != null)
				{
					stopMonitoring(selectedQueueName);
				}				
			}
		});
		stopButton.setEnabled(false);

	}
	/////////////// Source from the GetQueue Application Block /////////////
	/**
     * Retrieves a <code>QILQueue</code> instance.
     * <br>This method manages exception thrown by displaying an error message in the log window.
     * <br>This method uses the <code>GetQueue</code> Application Block's guidelines.
     * @param queueID ID of the queue.
     * @return instance of a queue or if not found or the connection is loss then a null is returned.
     */
    public QILQueue getQueue(String queueID) {
        try {
            return QILFactory.getQILFactory().getQueue(queueID);
        } catch (QILObjectNotFoundException e) {
        	sdkGui.writeLogMessage("Queue "+ queueID + " was not found. "+ e.toString(), OpenMediaSdkGui.errorStyle);    		
            return null;
        } catch (QILRequestFailedException e) {
        	sdkGui.writeLogMessage("Connection to Configuration Server may be lost. "+ e.toString(), OpenMediaSdkGui.errorStyle);   		
        	return null;
        }
    }
	/////////////// Source from the StartMonitoringQueue Application Block /////////////
	/**
	 * Starts monitoring a queue.
	 * <br>This method uses the <code>StartMonitoringQueue</code> Application Block's guidelines.
	 */
    public void startMonitoring(String queueName) {
        try {
            QILFactory.getQILFactory().getQueue(queueName).startMonitoring();
        } catch (QILUninitializedException __e) {
        	sdkGui.writeLogMessage("QILFactory is not initialized. "+__e.toString(), OpenMediaSdkGui.errorStyle);    		
        } catch (QILRequestFailedException __e) {
        	sdkGui.writeLogMessage("Connection to Configuration Server may be lost. "+__e.toString(), OpenMediaSdkGui.errorStyle);    		
        }
    }

    /**
	 * Stops monitoring a queue.
	 * <br>This method uses the <code>StartMonitoringQueue</code> Application Block's guidelines.
	 */
    public void stopMonitoring(String queueName) {
        try {
            QILFactory.getQILFactory().getQueue(queueName).stopMonitoring();
        } catch (QILUninitializedException __e) {
        	sdkGui.writeLogMessage("QILFactory is not initialized. "+__e.toString(), OpenMediaSdkGui.errorStyle);    		
        } catch (QILRequestFailedException __e) {
        	sdkGui.writeLogMessage("Connection to Configuration Server may be lost. "+__e.toString(), OpenMediaSdkGui.errorStyle);    		
        }
    }
    

	/**
	 * Receives QILQueueEvent events, which report changes in a queue.
	 * If you implement the QILQueueListener interface, you must implement this
	 * method.<br/><br/>
	 *
	 * This method uses a QueueEventThread to update the Queues
	 * table that lists available queues, and to log these QILQueueEvents as they occur.
	 * <br/><br/>
	 * This method's design conforms to the <code>MyQueueListener</code> Application Block.
	 */
	public void handleQueueEvent(QILQueueEvent event)
	{
		QueueEventThread p = new QueueEventThread(event);
		p.start();
	}
	/**
	 * Updates the Service table that lists available services, and to log these ServiceEvents as they occur.
	 */
	class QueueEventThread extends Thread
	{
		QILQueueEvent event;
		
		public QueueEventThread(QILQueueEvent _event)
		{
			event=_event;			
		}
		/////////////// Source from the MyQueueListener Application Block /////////////
		public void run()
		{
			if(event instanceof QILQueueMonitorStatusEvent)
			{
				//Getting the event
				QILQueueMonitorStatusEvent statusEvent = (QILQueueMonitorStatusEvent)event;
				QILQueue queue = statusEvent.getQueue();
				String queueName = queue.getID();
				//Updating GUI
				setQueueWidgets(queueName);
				
				//Creating a log message
				if(sdkGui.createDetailedQueueEventMessages)
				{
					String detailedLogMessage=queue.getID()
					+ " Reason("+statusEvent.getReason().toString()
					+ ") Old Status ["+statusEvent.getOldMonitorStatus().toString()
					+ "] ->  New Status ["+statusEvent.getNewMonitorStatus().toString()
					+ "] Annex["+queue.getAnnex().toString()+"]" ;
					sdkGui.writeLogMessage(detailedLogMessage, OpenMediaSdkGui.queueEventStyle);
					
				}else if (sdkGui.logQueueEvents)
				{
					String logMessage=queue.getID()+" "+statusEvent.getReason().toString()
					+ " New ("+statusEvent.getNewMonitorStatus().toString()+")";
					sdkGui.writeLogMessage(logMessage, OpenMediaSdkGui.queueEventStyle);
				}
				
			} else if(event instanceof QILQueueContentChangedEvent)
			{
				QILQueueContentChangedEvent changeEvent = (QILQueueContentChangedEvent)event;
				QILQueue queue = changeEvent.getQueue();
				String detailedLogMessage = queue.getID();
				String queueName = queue.getID();
				try{
					Iterator addedIxns = changeEvent.getAddedInteractions().iterator();
					Iterator deletedIxns = changeEvent.getRemovedInteractions().iterator();
					int interactionNumber = ((Integer) ixnPerQueue.get(queueName)).intValue(); 
					
					if(sdkGui.createDetailedQueueEventMessages)
					{
						//Getting Added Interactions
						while (addedIxns.hasNext())
						{
							QILInteraction ixn = (QILInteraction) addedIxns.next();
							interactionNumber++;
							detailedLogMessage+="\nAdded "+ixn.getID() +" "+ixn.getStatus().toString();
						}					
						//Getting Removed Interactions
						
						while (deletedIxns.hasNext())
						{
							QILInteraction ixn = (QILInteraction) deletedIxns.next();
							interactionNumber--;
							detailedLogMessage+="\nRemoved "+ixn.getID()+" "+ixn.getStatus().toString();
						}
						sdkGui.writeLogMessage(detailedLogMessage, OpenMediaSdkGui.queueEventStyle);
					}else if(sdkGui.logQueueEvents)
					{
						//					Getting Added Interactions
						while (addedIxns.hasNext())
						{
							QILInteraction ixn = (QILInteraction) addedIxns.next();
							interactionNumber++;
							detailedLogMessage+="\nAdded "+ixn.getID();
						}					
						//Getting Removed Interactions
						
						while (deletedIxns.hasNext())
						{
							QILInteraction ixn = (QILInteraction) deletedIxns.next();
							interactionNumber--;
							detailedLogMessage+="\nRemoved "+ixn.getID();
						}
						sdkGui.writeLogMessage(detailedLogMessage, OpenMediaSdkGui.queueEventStyle);
						ixnPerQueue.put(queueName, new Integer(interactionNumber)); 
						sdkGui.setQueue(queueName,  queue.isMonitored(),(new Integer(interactionNumber)).toString());	
						
					}
				}catch(Exception __e)
				{
					sdkGui.writeLogMessage("ContentChange: "+__e.getMessage(), OpenMediaSdkGui.errorStyle);
				}
			}
		}
	}
    /////////////////////////////////////////////////////////////
	/**
	 * Gets the queue object associated with the row selected in the Queues table.
	 * Updates the associated panels with queue information.
	 * This method is called by the SelectedQueueListener when the user selects a queue.
	 */
	public void handleQueueSelectionAt(int selectedRow) {
		selectedQueueName = sdkGui.getQueueNameAt(selectedRow);
		setQueueWidgets(selectedQueueName);
	}


	/**
	 * Updates the widgets associated with the selected queue (if needed).
	 */
	public void setQueueWidgets(String queueName)
	{
		QILQueue queue = this.getQueue(queueName);
		//Updating the Queues table
		if(queue != null)
		{
			if(!queue.isMonitored())
			{
				ixnPerQueue.put(queueName,new Integer(0));
			}
				
			sdkGui.setQueue(queueName,  queue.isMonitored(),ixnPerQueue.get(queueName).toString());	

			//Updating the Annex tree and the buttons if needed
			if(queueName.equals(selectedQueueName)){
				sdkGui.setAnnex(selectedQueueName, queue.getAnnex());
				if(queue.isMonitored()){
					startButton.setEnabled(false);
					stopButton.setEnabled(true);
				} else{
					startButton.setEnabled(true);
					stopButton.setEnabled(false);
				}
			}
		}	
	}
	
	/**
	 * Monitors the user selection in the <code>Queues</code> table.
	 * When the user selects a queue, this listener calls the {@link #handleQueueSelectionAt} method.     
	 */
	public class SelectedQueueListener implements ListSelectionListener {
		SimpleMonitorQueue application;

		public SelectedQueueListener(SimpleMonitorQueue application)
		{
			this.application = application;
		}

		public void valueChanged(ListSelectionEvent e) {
			//Ignore extra messages.
			if (e.getValueIsAdjusting()) return;

			ListSelectionModel lsm = (ListSelectionModel)e.getSource();
			if (!lsm.isSelectionEmpty()) {
				int selectedRow = lsm.getMinSelectionIndex();
				application.handleQueueSelectionAt(selectedRow);
			}
		}
	}
}