/*=====================================================================
Any authorized distribution of any copy of this code (including any related documentation) must reproduce the following restrictions, disclaimer and copyright notice:
The Genesys name, trademarks and/or logo(s) of Genesys shall not be used to name (even as a part of another name), endorse and/or promote products derived from this code without prior written permission from Genesys Telecommunications Laboratories, Inc.
The use, copy, and/or distribution of this code is subject to the terms of the Genesys Developer License Agreement.  This code shall not be used, copied, and/or distributed under any other license agreement.
THIS CODE IS PROVIDED BY GENESYS TELECOMMUNICATIONS LABORATORIES, INC. (GENESYS) AS IS WITHOUT ANY WARRANTY OF ANY KIND. GENESYS HEREBY DISCLAIMS ALL EXPRESS, IMPLIED, OR STATUTORY CONDITIONS, REPRESENTATIONS AND WARRANTIES WITH RESPECT TO THIS CODE (OR ANY PART THEREOF), INCLUDING, BUT NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT. GENESYS AND ITS SUPPLIERS SHALL NOT BE LIABLE FOR ANY DAMAGE SUFFERED AS A RESULT OF USING THIS CODE. IN NO EVENT SHALL GENESYS AND ITS SUPPLIERS BE LIABLE FOR ANY DIRECT, INDIRECT, CONSEQUENTIAL, ECONOMIC, INCIDENTAL, OR SPECIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, ANY LOST REVENUES OR PROFITS).
Copyright &copy; 2006 Genesys Telecommunications Laboratories, Inc. All rights reserved.
=====================================================================*/
/**
Queued Interaction Java API 7.5 Code Examples
*/
package queued.sdk.java.examples;

import com.genesyslab.omsdk.commons.*;
import com.genesyslab.omsdk.commons.event.ServiceListener;
import com.genesyslab.omsdk.commons.event.ServiceEvent;

import javax.swing.JFrame;
import javax.swing.SwingUtilities;

/**
 * Monitors connection services.
 * This example uses a GUI application to show how to monitor services:<ul>
 * <li>It displays the list of available services.
 * <li>It implements the ServiceListener interface to listen for changes in connections.
 * </ul>
 *
 * <p><strong>PLEASE NOTE:</strong> This code example will not work unless you
 * edit the configuration data in <code>OpenMediaSdk.properties</code> so that it matches
 * objects that already exist in your local Configuration Layer. <br><br>
 *
 * There are several steps needed to prepare an application to use the services
 * of the Queued Interaction Java API. This example uses the services of other
 * classes to carry out these functions: <br><br>
 *
 * <ul>
 * <li>{@link OpenMediaSdkData} makes various configuration data available to
 * the Queued SDK Java Library code examples</li>
 * <li>{@link OpenMediaSdkGui} creates the common graphical user interface
 * used by the code examples</li>
 * <li>{@link SimpleConnector} connects code examples to the Configuration Layer and
 * Interaction Server.</li>
 * </li>
 * </ul>
 * @version 7.5.001.00
 */
public class SimpleService  implements ServiceListener {

	/** User interface. */
	OpenMediaSdkGui sdkGui;
	/** Example type. */
	String exampleType;
	/** Connection to the Genesys servers. */
	SimpleConnector connection;

	/**
	 * Creates an instance of <code>SimpleService</code>.<br>
	 * @param windowTitle
	 *            The title of the GUI window for this example.
	 */
	public SimpleService(String windowTitle)
	{
		connection = new SimpleConnector();
		setExampleType();
		sdkGui = new OpenMediaSdkGui(windowTitle, exampleType);
        linkWidgetsToGui();
        sdkGui.mainGuiWindow.setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
        sdkGui.mainGuiWindow.addWindowListener(connection);
        sdkGui.mainGuiWindow.pack();
        sdkGui.mainGuiWindow.setVisible(true);

	}
	/**
	 * Links QIL information to GUI components.
	 * The <code>OpenMediaSdkGui</code> class creates a single graphical user interface
	 * that is used by each of the Queued Interaction Java code examples.
	 * This GUI contains all of the widgets used by the examples, but they are
	 * only placeholders. That is, they have no functionality when they
	 * are created in the GUI class. <br><br>
	 *
	 * When an example is run, the <code>linkWidgetsToGui()</code> method is used to
	 * associate these widgets with the appropriate fields in the example class,
	 * and to fill-in tables. <br><br>
	 *
	 * <code>sdkGui.setService(srv.getType().toString(),
						srv.getStatus().toString());</code> <br><br>
	 *
	 */
	public void linkWidgetsToGui()
	{
		try
		{
			//Getting service information
			ServiceInfo srv = OMSDKConnector.getInstance().getServiceInfo(ServiceType.CONFIGURATION);
			if(srv != null)
			{
				//Updating the table
				sdkGui.setService(srv.getType().toString(),
						srv.getStatus().toString());
				// Registering for this service
				// THIS IS AN IMPORTANT STEP:
				OMSDKConnector.addServiceListener(this,ServiceType.CONFIGURATION);
			}

			srv = OMSDKConnector.getInstance().getServiceInfo(ServiceType.INTERACTION_SERVER);
			if(srv != null)
			{
				//Updating the table
				sdkGui.setService(srv.getType().toString(),
						srv.getStatus().toString());
				//Registering for this service
				//THIS IS AN IMPORTANT STEP:
				OMSDKConnector.addServiceListener(this,ServiceType.INTERACTION_SERVER);
			}

		}catch(Exception exception){
			sdkGui.writeLogMessage(exception.toString(), OpenMediaSdkGui.errorStyle);
		}


	}

    /**
     * Indicates which example is running.
     * The <code>OpenMediaSdkGui</code> class creates a single graphical user interface
     * that is used by each of Open Media SDK Java Library code examples.
     * This GUI contains all of the widgets used by the examples -- but the class should enable only the
     * widgets used by a particular example. <br/><br/>
     *
     * This method tells the <code>OpenMediaSdkGui</code> class which example has called
     * the GUI, so that it can determine which widgets will be enabled when the
     * GUI is created.
     */
    public void setExampleType() {
        exampleType = "SimpleService";
    }

	public static void main(String[] args) {
		  //Schedule a job for the event-dispatching thread:
        //creating and showing this application's GUI.
        SwingUtilities.invokeLater(new Runnable() {
            public void run() {
            	SimpleService application = new SimpleService(
                        "Queued Interaction Java API: Simple Service");
            }
        });
	}

	/**
	 * Receives <code>ServiceEvent</code> events, which report changes in a connection service.
	 * If you implement the <code>ServiceListener</code> interface, you must implement this
	 * method.<br/><br/>
	 *
	 * The <code>SimpleService</code> example uses this method to update the Service
	 * table that lists available services, and to log these ServiceEvents as they occur.
	 * Event processing is performed in a dedicated ServiceEventThread thread class.
	 */
	public void handleServiceEvent(ServiceEvent event) {
		ServiceEventThread p = new ServiceEventThread(event);
		p.start();
	}
	
	/**
	 * Updates the Service table that lists available services, and to log these ServiceEvents as they occur.
	 */
	class ServiceEventThread extends Thread
	{
		ServiceEvent event;
		
		public ServiceEventThread(ServiceEvent _event)
		{
			event=_event;			
		}
		
		public void run()
		{
			ServiceInfo service = event.getServiceInfo();
			sdkGui.setService(service.getType().toString(),
					service.getStatus().toString());

	    	//Creating a log message for the event
	    	if(sdkGui.createDetailedServiceEventMessages)
			{
				String detailedLogMessage=event.getServiceType().toString()+ " "+event.getServiceInfo().getStatus().toString();
				sdkGui.writeLogMessage(detailedLogMessage, OpenMediaSdkGui.serviceEventStyle);
			}
	    	else if (sdkGui.logServiceEvents)
			{
				String logMessage=event.getServiceType().toString();
				sdkGui.writeLogMessage(logMessage, OpenMediaSdkGui.serviceEventStyle);

			}
		}
	}
}

