/*=====================================================================
Any authorized distribution of any copy of this code (including any related documentation) must reproduce the following restrictions, disclaimer and copyright notice:
The Genesys name, trademarks and/or logo(s) of Genesys shall not be used to name (even as a part of another name), endorse and/or promote products derived from this code without prior written permission from Genesys Telecommunications Laboratories, Inc.
The use, copy, and/or distribution of this code is subject to the terms of the Genesys Developer License Agreement.  This code shall not be used, copied, and/or distributed under any other license agreement.
THIS CODE IS PROVIDED BY GENESYS TELECOMMUNICATIONS LABORATORIES, INC. (GENESYS) AS IS WITHOUT ANY WARRANTY OF ANY KIND. GENESYS HEREBY DISCLAIMS ALL EXPRESS, IMPLIED, OR STATUTORY CONDITIONS, REPRESENTATIONS AND WARRANTIES WITH RESPECT TO THIS CODE (OR ANY PART THEREOF), INCLUDING, BUT NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT. GENESYS AND ITS SUPPLIERS SHALL NOT BE LIABLE FOR ANY DAMAGE SUFFERED AS A RESULT OF USING THIS CODE. IN NO EVENT SHALL GENESYS AND ITS SUPPLIERS BE LIABLE FOR ANY DIRECT, INDIRECT, CONSEQUENTIAL, ECONOMIC, INCIDENTAL, OR SPECIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, ANY LOST REVENUES OR PROFITS).
Copyright &copy; 2006 Genesys Telecommunications Laboratories, Inc. All rights reserved.
=====================================================================*/
/**
Queued Interaction Java API 7.5 Code Examples
@version 7.5.001.00
*/
package queued.sdk.java.examples;

import java.awt.event.ActionEvent;
import java.util.Collection;
import java.util.Iterator;

import javax.swing.AbstractAction;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.SwingUtilities;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import com.genesyslab.omsdk.qil.QILFactory;
import com.genesyslab.omsdk.qil.QILInteraction;
import com.genesyslab.omsdk.qil.QILOperationalMode;
import com.genesyslab.omsdk.qil.QILQueue;

import com.genesyslab.omsdk.qil.applicationblocks.monitoring.AdHocManagement;
import com.genesyslab.omsdk.qil.event.QILInteractionEvent;
import com.genesyslab.omsdk.qil.event.QILInteractionPropertiesChangedEvent;
import com.genesyslab.omsdk.qil.event.QILQueueContentChangedEvent;
import com.genesyslab.omsdk.qil.event.QILQueueEvent;
import com.genesyslab.omsdk.qil.exception.QILOperationalModeRestrictionException;
import com.genesyslab.omsdk.qil.exception.QILRequestFailedException;
import com.genesyslab.omsdk.qil.exception.QILUnsuccessfulModeChangeException;

/**
 * Provides Ad'Hoc Management features.
 * <p>This example provides perform supervisor actions on interactions available in monitored queues.
 * </p>
 * <p>Before you perform supervisor actions: <ul>
 * <li>Activate supervisor mode in the supervisor tab by clickilng the supervisor mode button.</li>
 * <li>Select an interaction in the interaction list of a queue.</li>
 * </ul>
 * </p>
 * <p>This code example uses the following application blocks:
 * <ul><li>com.genesyslab.omsdk.qil.applicationblocks.monitoring.AdHocManagement</li></ul></p>
 * @version 7.5.001.00
 * 
 */
public class SimpleSupervisor extends SimpleMonitorInteraction {

	AdHocManagement adHocManager;
	String selectedInteractionId;
	boolean managing = false;
    public JButton modeButton ;
    public JLabel modeLabel;
    public JLabel interactionIDLabel;
    public JTextField propNameTextField;
    public JTextField propValueTextField;
    
    public JButton propButton ;
    public JButton lockButton ;
    public JButton unlockButton ;
    public JButton pullButton ;
    public JButton leaveButton;    
    public JButton stopProcessingButton ;
	
	/**
	 * Constructor. Creates the <code>SimpleSupervisor</code> instance. 
	 * @param windowTitle
	 */
	public SimpleSupervisor(String windowTitle) {
		super(windowTitle);
		adHocManager = new AdHocManagement() ;
		sdkGui.activateIxnRowSelection(new SelectedInteractionListener(this));
		
	}

	/**
	 * Indicates which example is running.
	 * This method tells the <code>OpenMediaSdkGui</code> class which example has called
	 * the GUI, so that OpenMediaSdkGui can determine which widgets will be enabled when the
	 * GUI is created.
	 */
	public void setExampleType() {
		exampleType = "SimpleSupervisor";
	}
	
	/**
	 * Links QIL information to GUI components.
	 * The OpenMediaSdkGui class creates a single graphical user interface
	 * that is used by each of the Queued Interaction Java API code examples.
	 * This GUI contains all of the widgets used by the examples, but they are
	 * only placeholders. That is, they have no functionality when they
	 * are created in the GUI class. <br><br>
	 *
	 * When an example is run, the linkWidgetsToGui() method is used to
	 * associate these widgets with the appropriate fields in the example class,
	 * as shown here: <br><br>
	 *
	 * <code>startButton = sdkGui.startButton;</code> <br><br>
	 *
	 * In many cases, particularly for buttons and radio buttons, these fields
	 * are then linked to actions that correspond to the functions they are
	 * designed to carry out.
	 */
	public void linkWidgetsToGui() {
		super.linkWidgetsToGui();
		
		modeButton = sdkGui.modeButton;
		modeButton.setAction(new AbstractAction("Change Mode") {
			public void actionPerformed(ActionEvent actionEvent) {
				if(QILFactory.getQILFactory().getOperationalMode() == QILOperationalMode.REPORTING_MODE )
					try {
						adHocManager.changeOperationMode(true);
						modeLabel.setText("  Current Mode is: supervisor  ");
						propButton.setEnabled(true);
						lockButton.setEnabled(true);
						pullButton.setEnabled(true);
						stopProcessingButton.setEnabled(true);
						
					} catch (QILUnsuccessfulModeChangeException e) {
						sdkGui.writeLogMessage("On change mode: "+e.getMessage(), OpenMediaSdkGui.errorStyle);
					}
				else
					try {
						adHocManager.changeOperationMode(false);
						modeLabel.setText("  Current Mode is: reporting  ");
						propButton.setEnabled(false);
						lockButton.setEnabled(false);
						pullButton.setEnabled(false);
						stopProcessingButton.setEnabled(false);
					} catch (QILUnsuccessfulModeChangeException e1) {
						sdkGui.writeLogMessage("On change mode: "+e1.getMessage(), OpenMediaSdkGui.errorStyle);
						
					}
			}
		});
		
		
		
		propNameTextField = sdkGui.propNameTextField;;
		propValueTextField = sdkGui.propValueTextField;;
		    
		propButton = sdkGui.propButton;
		propButton.setAction(new AbstractAction("Change Properties") {
			public void actionPerformed(ActionEvent actionEvent) {
				try {
					QILFactory.getQILFactory().getInteractionManager().setProperty(selectedInteractionId,propNameTextField.getText(),propValueTextField.getSelectedText());
				} catch (QILOperationalModeRestrictionException e) {
					sdkGui.writeLogMessage("Change Property requested: "+e.getMessage(), OpenMediaSdkGui.errorStyle);
				} catch (QILRequestFailedException e) {
					sdkGui.writeLogMessage("Change Property requested: "+e.getMessage(), OpenMediaSdkGui.errorStyle);
				}
			}
		});
		propButton.setEnabled(false);
		
		lockButton = sdkGui.lockButton;
		lockButton.setAction(new AbstractAction("Lock") {
			public void actionPerformed(ActionEvent actionEvent) {
				if(selectedInteractionId!=null)
				{
					try {
						adHocManager.lock(selectedQueueName, selectedInteractionId,"","");
						lockButton.setEnabled(false);
						unlockButton.setEnabled(true);
						pullButton.setEnabled(false);
						stopProcessingButton.setEnabled(false);
						managing = true;
					} catch (QILOperationalModeRestrictionException e) {
						sdkGui.writeLogMessage("Lock requested, not authorized: "+e.getMessage(), OpenMediaSdkGui.errorStyle);
						
					} catch (QILRequestFailedException e) {
						sdkGui.writeLogMessage("Lock requested, failed: "+e.getMessage(), OpenMediaSdkGui.errorStyle);
						
					}
				}
			}
		});
		lockButton.setEnabled(false);
		
		unlockButton = sdkGui.unlockButton;
		unlockButton.setAction(new AbstractAction("Unlock") {
			public void actionPerformed(ActionEvent actionEvent) {
				if(selectedInteractionId!=null)
				{
					try {
						adHocManager.unlock(selectedInteractionId,"","");
						lockButton.setEnabled(true);
						unlockButton.setEnabled(false);
						pullButton.setEnabled(true);
						stopProcessingButton.setEnabled(true);
						managing= false;
					} catch (QILOperationalModeRestrictionException e) {
						// TODO Auto-generated catch block
						sdkGui.writeLogMessage("Unlock requested: "+e.getMessage(), OpenMediaSdkGui.errorStyle);
						
					} catch (QILRequestFailedException e) {
						sdkGui.writeLogMessage("Unlock requested: "+e.getMessage(), OpenMediaSdkGui.errorStyle);
						
					}
				}
			}
		});
		unlockButton.setEnabled(false);
		
		pullButton = sdkGui.pullButton;
		pullButton.setAction(new AbstractAction("Pull") {
			public void actionPerformed(ActionEvent actionEvent) {
				if(selectedInteractionId!=null)
				{
					try {
						adHocManager.pull(selectedInteractionId,"","");
						lockButton.setEnabled(false);
						leaveButton.setEnabled(true);
						pullButton.setEnabled(false);
						stopProcessingButton.setEnabled(false);
						managing = true;
					} catch (QILOperationalModeRestrictionException e) {
						sdkGui.writeLogMessage("Pull requested: "+e.getMessage(), OpenMediaSdkGui.errorStyle);
						
					} catch (QILRequestFailedException e) {
						sdkGui.writeLogMessage("Pull requested: "+e.getMessage(), OpenMediaSdkGui.errorStyle);
						
					}
				}
			}
		});
		pullButton.setEnabled(false);
		
		leaveButton= sdkGui.leaveButton;
		leaveButton.setAction(new AbstractAction("Leave") {
			public void actionPerformed(ActionEvent actionEvent) {
				if(selectedInteractionId!=null)
				{
					try {
						adHocManager.leave(selectedInteractionId,"","");
						lockButton.setEnabled(true);
						leaveButton.setEnabled(false);
						pullButton.setEnabled(true);
						stopProcessingButton.setEnabled(true);
						managing =false;
					} catch (QILOperationalModeRestrictionException e) {
						sdkGui.writeLogMessage("Leave requested: "+e.getMessage(), OpenMediaSdkGui.errorStyle);
						
					} catch (QILRequestFailedException e) {
						sdkGui.writeLogMessage("Leave requested: "+e.getMessage(), OpenMediaSdkGui.errorStyle);
						
					}
				}
			}
		});
		leaveButton.setEnabled(false);
		
		
		stopProcessingButton = sdkGui.stopProcessingButton;
		stopProcessingButton.setAction(new AbstractAction("Stop") {
			public void actionPerformed(ActionEvent actionEvent) {
				if(selectedInteractionId!=null)
				{
					try {
						adHocManager.stopProcessing(selectedInteractionId,"","");
						lockButton.setEnabled(true);
						leaveButton.setEnabled(false);
						pullButton.setEnabled(true);
						stopProcessingButton.setEnabled(false);
						managing =false;
					} catch (QILOperationalModeRestrictionException e) {
						sdkGui.writeLogMessage("Leave requested: "+e.getMessage(), OpenMediaSdkGui.errorStyle);
						
					} catch (QILRequestFailedException e) {
						sdkGui.writeLogMessage("Leave requested: "+e.getMessage(), OpenMediaSdkGui.errorStyle);
						
					}
				}
			}
		});
		stopProcessingButton.setEnabled(false);
		
		
		modeLabel = sdkGui.modeLabel;
		interactionIDLabel = sdkGui.interactionIDLabel;
		
		if(QILFactory.getQILFactory().getOperationalMode() == QILOperationalMode.REPORTING_MODE )
		{
			propButton.setEnabled(false);
			lockButton.setEnabled(false);
			pullButton.setEnabled(false);
			stopProcessingButton.setEnabled(false);
			modeLabel.setText("  Current Mode is: reporting  ");
			managing = false;
			this.sdkGui.clearIxnProperties();
			interactionIDLabel.setText("ID : ");
			this.selectedInteractionId = null;
		}
		else
		{
			propButton.setEnabled(true);
			lockButton.setEnabled(true);
			pullButton.setEnabled(true);
			stopProcessingButton.setEnabled(true);
			modeLabel.setText("  Current Mode is: supervisor  ");
		}
		
	}
	/**
	 * Handles the interaction selection in the interaction list.
	 */
	public void handleInteractionSelectionAt(int selectedRow)
	{
		if(!managing)
		{
			try{
				
				this.selectedInteractionId = this.sdkGui.getInteractionNameAt(this.selectedQueueName, selectedRow);
				if(this.selectedInteractionId != null)
				{
					interactionIDLabel.setText("ID: "+selectedInteractionId);
					this.sdkGui.switchPropertyTree(selectedQueueName,selectedInteractionId);
				}
			}catch(Exception __e)
			{
				sdkGui.writeLogMessage("Interaction selection failed. "+__e.getMessage(), OpenMediaSdkGui.errorStyle);
				
			}
		}
	}
	
				
	/**
	 * Manages interaction events to update the interaction property tree when needed.
	 * Event processing is performed through the InteractionPropertiesChangedThread class. 
	 * @see com.genesyslab.omsdk.qil.event.QILInteractionListener#handleInteractionEvent(com.genesyslab.omsdk.qil.event.QILInteractionEvent)
	 */
	public void handleInteractionEvent(QILInteractionEvent event) {
		
		super.handleInteractionEvent(event);
		InteractionPropertiesChangedThread p = new InteractionPropertiesChangedThread(event);
		p.start();
		
	}
	
	class InteractionPropertiesChangedThread extends Thread
	{
		QILInteractionEvent event;
		
		public InteractionPropertiesChangedThread(QILInteractionEvent _event)
		{
			event=_event;	
		}
		
		public void run()
		{
			if (event instanceof QILInteractionPropertiesChangedEvent) {
				QILInteractionPropertiesChangedEvent eventPropertiesChanged = (QILInteractionPropertiesChangedEvent) event;
				if(eventPropertiesChanged.getInteraction().getID().equals(selectedInteractionId))
				{
					sdkGui.switchPropertyTree(selectedQueueName,selectedInteractionId);
				}
			}
		}
	}
	
	
	
	/**
	 * Updates interaction information.
	 * @param queueName
	 * @param addedInteractions
	 * @param removedInteractions
	 */
	public void setIxnPropertiesWidgets(String queueName, Collection removedInteractions)			
	{
		//Getting Removed Interactions
		Iterator deletedIxns = removedInteractions.iterator();
		while (deletedIxns.hasNext())
		{
			QILInteraction ixn = (QILInteraction) deletedIxns.next();
			sdkGui.removeIxnProperties(queueName, ixn.getID());
			if(ixn.getID().equals(selectedInteractionId) && !this.managing)
				this.sdkGui.clearIxnProperties();
		}
	
	}
	/**
	 * Handles queue events.
	 * This method overloads the parent <code>SimpleMonitorQueue.handleQueueEvent()</code> method to listen the
	 * interaction events of a monitored queue. It also updates the interaction properties widgets associated with the queue.
	 */
	public void handleQueueEvent(QILQueueEvent event)
	{
		super.handleQueueEvent(event);
		RemovedInteractionThread p = new RemovedInteractionThread(event);
		p.start();
		
	}
	
	class RemovedInteractionThread extends Thread
	{
		QILQueueEvent event;
		
		public RemovedInteractionThread(QILQueueEvent _event)
		{
			event=_event;	
		}
		
		public void run()
		{
			if(event instanceof QILQueueContentChangedEvent)
			{
				QILQueueContentChangedEvent changeEvent = (QILQueueContentChangedEvent)event;
				QILQueue queue = changeEvent.getQueue();			
				setIxnPropertiesWidgets(queue.getID(), changeEvent.getRemovedInteractions());			
			}
		}
	}

	
	
	/**
	 * Monitors the user selection in the <code>Queues</code> table.
	 * When the user selects a queue, this listener calls the {@link #handleQueueSelectionAt} method.   
	 */
	public class SelectedInteractionListener implements ListSelectionListener {
		SimpleSupervisor application;

		public SelectedInteractionListener(SimpleSupervisor application)
		{
			this.application = application;
		}

		public void valueChanged(ListSelectionEvent e) {
			//Ignore extra messages.
			if (e.getValueIsAdjusting()) return;

			ListSelectionModel lsm = (ListSelectionModel)e.getSource();
			if (!lsm.isSelectionEmpty()) {
				int selectedRow = lsm.getMinSelectionIndex();
				application.handleInteractionSelectionAt(selectedRow);
			}
		}
	}
	
	public static void main(String[] args) {
//		Schedule a job for the event-dispatching thread:
		//creating and showing this application's GUI.
		SwingUtilities.invokeLater(new Runnable() {
			public void run() {
				SimpleSupervisor application = new SimpleSupervisor(
				"Queued Interaction Java API: Simple Supervisor");
			}
		});
	}
}
