/*=====================================================================

Any authorized distribution of any copy of this code (including any related documentation) must reproduce the following restrictions, disclaimer and copyright notice:

The Genesys name, trademarks and/or logo(s) of Genesys shall not be used to name (even as a part of another name), endorse and/or promote products derived from this code without prior written permission from Genesys Telecommunications Laboratories, Inc.

The use, copy, and/or distribution of this code is subject to the terms of the Genesys Developer License Agreement.  This code shall not be used, copied, and/or distributed under any other license agreement.

THIS CODE IS PROVIDED BY GENESYS TELECOMMUNICATIONS LABORATORIES, INC. (GENESYS) AS IS WITHOUT ANY WARRANTY OF ANY KIND. GENESYS HEREBY DISCLAIMS ALL EXPRESS, IMPLIED, OR STATUTORY CONDITIONS, REPRESENTATIONS AND WARRANTIES WITH RESPECT TO THIS CODE (OR ANY PART THEREOF), INCLUDING, BUT NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT. GENESYS AND ITS SUPPLIERS SHALL NOT BE LIABLE FOR ANY DAMAGE SUFFERED AS A RESULT OF USING THIS CODE. IN NO EVENT SHALL GENESYS AND ITS SUPPLIERS BE LIABLE FOR ANY DIRECT, INDIRECT, CONSEQUENTIAL, ECONOMIC, INCIDENTAL, OR SPECIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, ANY LOST REVENUES OR PROFITS).

Copyright &copy; 2008 Genesys Telecommunications Laboratories, Inc. All rights reserved.

=====================================================================*/

package com.genesyslab.gis.services.statistic;

import com.genesyslab.gis.services.session.ConnectSessionServiceExample;
import com.genesyslab.gis.services.statistic.types.Metric;
import com.genesyslab.gis.services.statistic.types.ObjectIdType;
import com.genesyslab.gis.services.statistic.types.ObjectType;
import com.genesyslab.gis.services.statistic.types.RetrieveStatisticResponse;
import com.genesyslab.gis.services.statistic.types.Schedule;
import com.genesyslab.gis.services.statistic.types.ScheduleMode;
import com.genesyslab.gis.services.statistic.types.Statistic;
import com.genesyslab.gis.services.statistic.types.StatisticType;
import com.genesyslab.gis.services.statistic.types.StatisticValue;
import com.genesyslab.gis.services.statistic.types.TimeInterval;
import com.genesyslab.gis.services.statistic.types.TimeIntervalType;
import com.genesyslab.gis.util.PropertiesLoader;

/*
 * This example demonstrates how to retrieve a statistic without a subscription.
 * Most of the code in here is similar to SubscribeToStatisticExample.java; both
 * examples construct the Statistic object.
 * 
 * This is a one time retrieval. If you need to retrieve a statistic frequently,
 * see RetrieveSubscribedStatisticExample.java.
 * 
 * @Date June, 2008
 */

public class RetrieveStatisticExample {

	public StatServiceService statService;
	public StatService statPort;

	public RetrieveStatisticExample() {
	}

	/*
	 * This method uses the ConnectToServiceExample to login and retrieve the
	 * session id. Then it subscribes, that is request for license, to a
	 * statistic. The statistic subscribed is for an agent's total login time.
	 * 
	 * @param gisServer String value in format of <gisHost>:<gisPort>
	 */
	public void retrieve(String gisServer) throws Exception {

		String[] services = { "GIS_STATSERVICE" };

		// first step is to connect to a service. See
		// ConnectToServiceExample.java in current folder.
		ConnectSessionServiceExample cs = new ConnectSessionServiceExample(
				services);
		cs.execute(gisServer);

		// Make a service
		statService = new StatServiceServiceLocator();

		// Now use the service to get a stub which implements the SDI.
		statPort = statService.getStatService(new java.net.URL("http://"
				+ gisServer + "/gis/services/StatService?GISsessionId="
				+ cs.sid));

		// build a statistic object
		Statistic statistic = new Statistic();
		statistic.setStatisticId("statref1");

		// building object id
		ObjectIdType objectId = new ObjectIdType();
		String agentId = PropertiesLoader.getOption("statistic.agentId");
		System.out.println("Statistic needed for "+agentId);
		// enter the employee id for this method
		objectId.setId(agentId);
		// The agent object is under the Tenant statistic.tenant. Note statistic.tenant is the tenant name
		// defined in gis.properties file
		String tenant = PropertiesLoader.getOption("statistic.tenant");
		objectId.setTenantName(tenant);

		// adding object id data to statistic
		statistic.setObjectId(objectId);

		// building metric data
		Metric metric = new Metric();
		StatisticType statisticType = new StatisticType();
		ObjectType[] objectType = new ObjectType[] { ObjectType.Agent };
		statisticType.setObjectType(objectType);
		metric.setStatisticType(statisticType);
		System.out.println("Statistic of interest is agent's total login time");
		metric.setTypeName("TotalLoginTime");
		TimeInterval timeInterval = new TimeInterval();
		timeInterval.setIntervalType(TimeIntervalType.GrowingWindow);
		metric.setInterval(timeInterval);

		// adding metric data to statistic
		statistic.setMetric(metric);

		// specify statistic schedule
		Schedule schedule = new Schedule();
		schedule.setNotificationMode(ScheduleMode.ChangesBased);

		// adding schedule data to statistic
		statistic.setSchedule(schedule);

		// Subscribe to the statistic. Note statistic.statserver.name is the application name
		// of a particular stat server defined in gis.properties file
		String statServeName = PropertiesLoader.getOption("statistic.statserver.name");
		RetrieveStatisticResponse retStat = statPort.retrieveStatistic(
				statistic, statServeName);
		// the first time the example calls this method in GIS will only
		// initialize the statistic. The value returned is 0.
		System.out.println("Statistic retrieved:");
		StatisticValue stat = retStat.getStatisticValue();
		System.out.print("Initializing the statistic...");
		System.out.println("value retrieved is = "
				+ stat.getEventValues()[0].getLValue());

		// The sleep method here creates a 32 second delay before the second
		// call.
		try {
			Thread.sleep(32000);
		} catch (InterruptedException interruptedexception) {
		}

		// the second call retrieves the statistic.
		retStat = statPort.retrieveStatistic(statistic, statServeName);
		stat = retStat.getStatisticValue();
		System.out.print("Retrieving statistic...");
		System.out.println("value retrieved is = "
				+ stat.getEventValues()[0].getLValue());
	}

	// Standalone option to test run this example. Creates a new instance of
	// this object and starts the process
	public static void main(String[] args) {
		String gisHost = PropertiesLoader.getOption("gis.host");
		String gisPort = PropertiesLoader.getOption("gis.port");
		try {
			RetrieveStatisticExample sb = new RetrieveStatisticExample();
			sb.retrieve(gisHost + ":" + gisPort);

		} catch (Exception serviceException) {
			System.out.println(serviceException.getMessage());
		}
	}
}