package sample;

import com.genesyslab.platform.management.protocol.LocalControlAgentProtocol;
import com.genesyslab.platform.management.protocol.ApplicationStatus;
import com.genesyslab.platform.management.protocol.ApplicationExecutionMode;

import com.genesyslab.platform.management.protocol.localcontrolagent.requests.RequestSCSConnectionStatus;
import com.genesyslab.platform.management.protocol.localcontrolagent.requests.RequestUpdateStatus;
import com.genesyslab.platform.management.protocol.localcontrolagent.events.EventChangeExecutionMode;
import com.genesyslab.platform.management.protocol.localcontrolagent.events.EventHostUnderControl;
import com.genesyslab.platform.management.protocol.localcontrolagent.events.EventInitHeartbeats;
import com.genesyslab.platform.management.protocol.localcontrolagent.events.EventNoControllingSCS;
import com.genesyslab.platform.management.protocol.localcontrolagent.events.EventSuspendApplication;
import com.genesyslab.platform.management.protocol.localcontrolagent.responses.ResponseExecutionModeChanged;

import com.genesyslab.platform.commons.protocol.ChannelClosedEvent;
import com.genesyslab.platform.commons.protocol.ChannelErrorEvent;
import com.genesyslab.platform.commons.protocol.ChannelListener;
import com.genesyslab.platform.commons.protocol.ChannelState;
import com.genesyslab.platform.commons.protocol.Message;
import com.genesyslab.platform.commons.protocol.MessageHandler;
import com.genesyslab.platform.commons.protocol.ProtocolException;
import com.genesyslab.platform.commons.protocol.Referenceable;

import com.genesyslab.platform.commons.timer.TimerFactory;
import com.genesyslab.platform.commons.timer.TimerAction;
import com.genesyslab.platform.commons.timer.TimerActionTicket;
import com.genesyslab.platform.commons.threading.SingleThreadInvoker;

import java.util.EventObject;


/**
 * Sample "application" to wrap LCA connection handling its base requests, events and state report messages.
 * <p/>
 * It declares protected abstract methods to be implemented by user application:<ul>
 * <li>{@link #onLCAConnectionOpened()} - to notify application about LCA connection open;</li>
 * <li>{@link #onLCAConnectionLost(ChannelClosedEvent)} - to notify application about LCA connection closed/lost;</li>
 * <li>{@link #onChangeExecutionMode(ApplicationExecutionMode)} - to notify application
 *         that its execution mode should be changed to <code>"Primary"</code>,
 *         <code>"Backup"</code> or <code>"Exiting"</code>;</li>
 * <li>{@link #onSuspendApplication()} - for application suspend (first part of "Graceful Stop");</li>
 * <li>{@link #onHostUnderSCSControl(EventHostUnderControl)} - to notify about live connection from LCA to SCS;</li>
 * <li>{@link #onHostOffSCSControl(EventNoControllingSCS)} - to notify about absence of connection from LCA to SCS.</li>
 * </ul>
 */
public abstract class AbstractServerApp {

    private LocalControlAgentProtocol lcaProtocol = null;

    /**
     * Creates and initializes LCA channel with its listener and message handler.
     * And asynchronously initiates connection open procedure.<br/>
     *
     * Once the connection is opened, LCA may send queries to change execution mode, exit, suspend, etc.
     * So, application should be ready to handle it.<br/>
     *
     * Incoming LCA messages will be processed with dedicated LCA invoker thread.
     *
     * @param lcaPortNum port number of the LCA
     * @param clientType type of this client application
     * @param clientName name of this client application
     * @param clientId DBID of this client application
     * @throws ProtocolException
     * @see LocalControlAgentProtocol
     */
    protected void openLCA(
            final int     lcaPortNum,
            final int     clientType,
            final String  clientName,
            final Integer clientId)
                throws ProtocolException {

        if (lcaProtocol != null) {
            throw new ProtocolException("The protocol is already created");
        }

        lcaProtocol = new LocalControlAgentProtocol(lcaPortNum);

        lcaProtocol.setApplicationType(clientType);
        if (clientName != null) {
            lcaProtocol.setClientName(clientName);
        }
        if (clientId != null) {
            lcaProtocol.setClientId(clientId);
        }

        lcaProtocol.addChannelListener(new LCAChannelListener());
        lcaProtocol.setMessageHandler(new LCAMessageHandler());
        lcaProtocol.setInvoker(new SingleThreadInvoker("LCA"));

        lcaProtocol.beginOpen();
    }

    /**
     * Closes the LCA protocol connection.
     *
     * @throws ProtocolException
     * @throws InterruptedException
     */
    protected void closeLCA() throws ProtocolException, InterruptedException {
        if (lcaProtocol != null) {
            try {
                lcaProtocol.close();
            } finally {
                lcaProtocol = null;
            }
        }
    }


    public boolean isLCAConnected() {
        if (lcaProtocol != null) {
            return (lcaProtocol.getState() == ChannelState.Opened);
        }
        return false;
    }

    public ApplicationExecutionMode getExecutionMode() {
        if (lcaProtocol != null) {
            return lcaProtocol.getExecutionMode();
        }
        return null;
    }

    public Integer getControlStatus() {
        if (lcaProtocol != null) {
            return lcaProtocol.getControlStatus();
        }
        return null;
    }


    /**
     * This method is called to send a response for MF request {@link EventChangeExecutionMode}.
     * <p/>
     * It is called by internal message handler and there is no need
     * to additionally call it from somewhere else, so, it's "private".
     *
     * @param referenceId reference ID of the MF mode change request
     * @see ResponseExecutionModeChanged
     * @see EventChangeExecutionMode
     */
    private void doResponseExecutionModeChanged(
            final Integer referenceId) {
        ResponseExecutionModeChanged theResponse = ResponseExecutionModeChanged.create();
        if (referenceId != null) {
            theResponse.setReferenceId(referenceId);
        }

        if (lcaProtocol.getProcessId() > 0) {
            theResponse.setProcessId(lcaProtocol.getProcessId());
        }
        theResponse.setExecutionMode(lcaProtocol.getExecutionMode());

        try {
            lcaProtocol.send(theResponse);
        } catch (Exception e) {
            // log exception / throw some other one
        }
    }

    /**
     * Protected method for the application status update in the Genesys Management Framework.
     *
     * @param execMode execution (mode or null)
     * @param controlStatus actual application control status
     * @see RequestUpdateStatus
     */
    protected void doUpdateStatus(
            final ApplicationExecutionMode execMode,
            final ApplicationStatus controlStatus) {

        if (controlStatus == null) {
            throw new NullPointerException("controlStatus");
        }

        RequestUpdateStatus theRequest = RequestUpdateStatus.create();
        theRequest.setReferenceId(0);

        if (lcaProtocol.getClientName() != null) {
            theRequest.setApplicationName(lcaProtocol.getClientName());
        }
        if (lcaProtocol.getClientId() != 0) {
            theRequest.setControlObjectId(lcaProtocol.getClientId());
        }

        if (execMode != null) {
            theRequest.setExecutionMode(execMode);
        }
        theRequest.setControlStatus(controlStatus.ordinal());
        if (lcaProtocol.getProcessId() != -1) {
            theRequest.setProcessId(lcaProtocol.getProcessId());
        }

        try {
            lcaProtocol.setControlStatus(controlStatus.ordinal());
            if (execMode != null) {
                lcaProtocol.setExecutionMode(execMode);
            }

            lcaProtocol.send(theRequest);
        } catch (Exception e) {
            // log exception / throw some other one
        }
    }

    /**
     * This method sends specific request to LCA ("RequestSCSConnectionStatus")
     * to get state of LCA to SCS connection.<br/>
     * It causes LCA to send back "EventHostUnderControl" or "EventHostOffSCSControl". As a result,
     * {@link #onHostUnderSCSControl(EventHostUnderControl)} or {@link #onHostOffSCSControl(EventNoControllingSCS)}
     * method will be executed (with LCA invoker thread).
     *
     * <p><b>Note:</b> This feature is new for MF 8.1.2. Earlier LCA versions do not recognize these messages,
     * so there will be no effect on this method call - just log record in the LCA log (if its configured).
     *
     * @see #onHostUnderSCSControl(EventHostUnderControl)
     * @see #onHostOffSCSControl(EventNoControllingSCS)
     * @see RequestSCSConnectionStatus
     */
    protected void doRequestSCSConnectionStatus() {
        try {
            lcaProtocol.send(RequestSCSConnectionStatus.create());
        } catch (Exception e) {}
    }


    /**
     * This method will be called to notify application about successful connection to LCA.
     *
     * @see #onLCAConnectionLost(ChannelClosedEvent)
     */
    protected abstract void onLCAConnectionOpened();

    /**
     * This method will be called to notify application closed or lost connection to LCA.<br/>
     * It may also be called in case of connection failure (exception on connection open procedure).<br/>
     * This sample contains simple connection restore logic, which, in case of connection lost,
     * schedules automatic reconnection with 30 seconds delay. So, this event may raise multiple times
     * meaning that we still have no connection to the LCA.
     *
     * @param event channel closed event with reason
     * @see #onLCAConnectionOpened()
     */
    protected abstract void onLCAConnectionLost(ChannelClosedEvent event);


    /**
     * This method will be called to notify application about SCS-to-LCA connection availability.
     *
     * @param event the protocol event
     * @see #onHostOffSCSControl(EventNoControllingSCS)
     */
    protected abstract void onHostUnderSCSControl(EventHostUnderControl event);

    /**
     * This method will be called to notify application about absence of SCS-to-LCA connection.
     *
     * @param event the protocol event
     * @see #onHostUnderSCSControl(EventHostUnderControl)
     */
    protected abstract void onHostOffSCSControl(EventNoControllingSCS event);


    /**
     * This is a callback method initiated by SCS to switch application to new execution mode.
     *
     * @param execMode the protocol event
     * @return <code>true</code> if mode change has been finished successfully
     * @see EventChangeExecutionMode
     */
    protected abstract boolean onChangeExecutionMode(
            ApplicationExecutionMode execMode);

    /**
     * This is a callback method initiated by SCS to request application suspend.<br/>
     * It is a first step in the "Graceful Stop" Management Framework feature.<br/>
     *
     * @see #onChangeExecutionMode(ApplicationExecutionMode)
     * @see EventSuspendApplication
     */
    protected abstract void onSuspendApplication();


    protected void onLCAConnectionError(final ChannelErrorEvent event) {
        // do nothing (by default)
    }


    private TimerActionTicket lcaReopenTicket = null;

    /**
     * Internal channel state listener for the LCA protocol connection.
     */
    private class LCAChannelListener implements ChannelListener {

        public void onChannelOpened(final EventObject event) {
            onLCAConnectionOpened();
        }

        public void onChannelClosed(final ChannelClosedEvent event) {
            onLCAConnectionLost(event);

            if ((event != null)
                    && (event.getCause() != null)
                    && (getExecutionMode() != ApplicationExecutionMode.Exiting)
                    && (lcaReopenTicket == null)) {
                lcaReopenTicket = TimerFactory.getTimer().schedule(
                        30000, // 30 seconds
                        new LCAReopenTask());
            }
        }

        public void onChannelError(final ChannelErrorEvent event) {
            onLCAConnectionError(event);
        }
    }


    /**
     * Internal message handler for the LCA protocol connection.
     */
    private class LCAMessageHandler implements MessageHandler {

        public void onMessage(final Message message) {

            if (message != null) { try {
                Integer referenceId = null;
                if (message instanceof Referenceable) {
                    referenceId = (Integer) ((Referenceable) message).retreiveReference();
                }

                switch (message.messageId()) {

                case EventChangeExecutionMode.ID:
                    ApplicationExecutionMode newMode = ((EventChangeExecutionMode) message).getExecutionMode();
                    try {
                        if (onChangeExecutionMode(newMode)) {
                            lcaProtocol.setExecutionMode(newMode);
                        }
                    } finally {
                        doResponseExecutionModeChanged(referenceId);
                    }
                    break;

                case EventSuspendApplication.ID:
                    onSuspendApplication();
                    doUpdateStatus(getExecutionMode(), ApplicationStatus.Suspended);
                    break;

                case EventHostUnderControl.ID:
                    onHostUnderSCSControl((EventHostUnderControl) message);
                    break;

                case EventNoControllingSCS.ID:
                    onHostOffSCSControl((EventNoControllingSCS) message);
                    break;

                case EventInitHeartbeats.ID:
                    // Nothing special is required here (in general) -
                    // the protocol handles the heartbeats internally.
                    break;

                default:
                    // report/log non-handled LCA message
                    break;
                }

            } catch(Exception e) {
                // log exception / throw some other one
            }}
        }
    }


    /**
     * Timer task, which may be scheduled in case of LCA connection lost, to start reopen procedure later.<br/>
     * It's used internally.
     */
    private class LCAReopenTask implements TimerAction {
        public void onTimer() {
            if (lcaReopenTicket != null) {
                lcaReopenTicket.cancel();
                lcaReopenTicket = null;
            }
            if ((lcaProtocol != null)
                    && (lcaProtocol.getState() != ChannelState.Opened)
                    && (getExecutionMode() != ApplicationExecutionMode.Exiting)) {
                try {
                    lcaProtocol.beginOpen();
                } catch (ProtocolException e) {}
            }
        }
    }
}
