﻿using System;
using Genesyslab.Platform.Commons.Connection.Timer;
using Genesyslab.Platform.Commons.Protocols;
using Genesyslab.Platform.Commons.Threading;
using Genesyslab.Platform.Management.Protocols;
using Genesyslab.Platform.Management.Protocols.LocalControlAgent.Events;
using Genesyslab.Platform.Management.Protocols.LocalControlAgent.Requests;
using Genesyslab.Platform.Management.Protocols.LocalControlAgent.Responses;

namespace SampleServerApp
{

    /// <summary>
    /// This class wraps comminication with Local Control Agebt and it's state reporting to Genesys Management Framework.
    /// It declares protected abstract methods to be implemented by user application:
    /// <list type="bullet">
    /// <item> 
    /// <description><see cref="OnLCAConnectionOpened()"/> - to notify application about LCA connection open</description> 
    /// </item> 
    /// <item> 
    ///<description><see cref="OnLCAConnectionLost(EventArgs)"/>  - to notify application about LCA connection closed/lost</description> 
    /// </item> 
    /// <item> 
    /// <description>
    ///         <see cref="OnChangeExecutionMode(ApplicationExecutionMode)"/> - to notify application 
    ///         that its execution mode should be changed to "Primary", "Backup" or "Exiting"
    /// </description> 
    /// </item> 
    /// <item> 
    /// <description><seealso cref="OnSuspendApplication()"/> - for application suspend (first part of "Graceful Stop")</description> 
    /// </item> 
    /// <item> 
    /// <description><seealso cref="OnHostUnderSCSControl(EventHostUnderControl)"/> - to notify about live connection from LCA to SCS</description> 
    /// </item> 
    /// <item> 
    /// <description><seealso cref="OnHostOffSCSControl(EventNoControllingSCS)"/> - to notify about absence of connection from LCA to SCS.</description> 
    /// </item> 
    /// </list> 
    /// </summary>
    public abstract class AbstractServerApp
    {

        private LocalControlAgentProtocol lcaProtocol = null;

        /// <summary>
        /// Creates and initializes connection with Local Control Agent. Subscribes to the protocol events.     
        /// 
        /// Once the connection is opened, LCA may send queries to change execution mode, exit, suspend, etc.
        /// So, application should be ready to handle it.<br/>
        /// 
        /// Incoming LCA messages will be processed with dedicated LCA invoker thread.
        /// </summary>
        /// <param name="lcaPort">Local Control Agent port</param> 
        /// <param name="clientType">type of this client application</param> 
        /// <param name="clientName">name of this client applicationt</param> 
        /// <param name="clientId">DBID of this client application</param> 
        /// <seealso cref="LocalControlAgentProtocol"/>        
        protected void OpenLCA(int lcaPort, int clientType, String clientName, int? clientId)
        {

            if (lcaProtocol != null)
            {
                throw new ProtocolException("The protocol is already created");
            }

            lcaProtocol = new LocalControlAgentProtocol(lcaPort);

            lcaProtocol.ApplicationType = clientType;
            if (clientName != null)
            {
                lcaProtocol.ClientName = clientName;
            }
            if (clientId != null)
            {
                lcaProtocol.ClientId = (int) clientId;
            }

            lcaProtocol.ControlStatus = (int)ApplicationStatus.Initializing;

            lcaProtocol.Opened += OnChannelOpened;
            lcaProtocol.Closed += OnChannelClosed;
            lcaProtocol.Error += OnChannelError;

            lcaProtocol.Received += OnLCAaMessage;

            //Set invoker that will be used to call user's callbacks for events like Opened, Closed, Received, Error.
            lcaProtocol.Invoker = new SingleThreadInvoker("LCA");

            lcaProtocol.BeginOpen();
        }

        /// <summary>
        /// Closes connection with Local Control Agent.
        /// </summary>
        protected void CloseLCA()
        {
            if (lcaProtocol != null)
            {
                try
                {
                    lcaProtocol.Close();
                }
                finally
                {
                    lcaProtocol.Opened -= OnChannelOpened;
                    lcaProtocol.Closed -= OnChannelClosed;
                    lcaProtocol.Error -= OnChannelError;
                    lcaProtocol = null;
                }
            }
        }


        public bool LCAConnected
        {
            get
            {
                if (lcaProtocol != null)
                {
                    return (lcaProtocol.State == ChannelState.Opened);
                }
                return false;
            }
        }

        public ApplicationExecutionMode? ExecutionMode
        {
            get
            {
                if (lcaProtocol != null)
                {
                    return lcaProtocol.ExecutionMode;
                }
                return null;
            }
        }

        public int? ControlStatus
        {
            get
            {
                if (lcaProtocol != null)
                {
                    return lcaProtocol.ControlStatus;
                }
                return null;
            }
        }


        /// <summary>
        /// This method sends a response for  Managemnt Framework request. 
        /// It is called by internal message handler and there is no need
        /// to additionally call it from somewhere else, so, it's "private".   
        /// </summary>
        /// <param name="referenceId">The reference ID of the request</param>
        /// <seealso cref="ResponseExecutionModeChanged"/>
        /// <seealso cref="EventChangeExecutionMode"/>

        private void DoResponseExecutionModeChanged(int referenceId)
        {
            ResponseExecutionModeChanged theResponse = ResponseExecutionModeChanged.Create();

            theResponse.ReferenceId = referenceId;

            if (lcaProtocol.ProcessId > 0)
            {
                theResponse.ProcessId = lcaProtocol.ProcessId;
            }
            theResponse.ExecutionMode = lcaProtocol.ExecutionMode;

            try
            {
                lcaProtocol.Send(theResponse);
            }
            catch (Exception e)
            {
                // log exception / throw some other one
            }
        }

        /// <summary>
        /// Protected method for the application status update in the Genesys Management Framework.
        /// </summary>
        /// <param name="execMode">Actual application execution mode</param> 
        /// <param name="controlStatus">Actual application control status</param> 
        /// <seealso cref="RequestUpdateStatus"/>
        protected void DoUpdateStatus(ApplicationExecutionMode? execMode, ApplicationStatus controlStatus)
        {

            RequestUpdateStatus theRequest = RequestUpdateStatus.Create();
            theRequest.ReferenceId = 0;

            if (lcaProtocol.ClientName != null)
            {
                theRequest.ApplicationName = lcaProtocol.ClientName;
            }
            if (lcaProtocol.ClientId != 0)
            {
                theRequest.ControlObjectId = lcaProtocol.ClientId;
            }

            if (execMode != null)
            {
                theRequest.ExecutionMode = (ApplicationExecutionMode) execMode;
            }

            theRequest.ControlStatus = (int) controlStatus;
            if (lcaProtocol.ProcessId != -1)
            {
                theRequest.ProcessId = lcaProtocol.ProcessId;
            }

            try
            {
                lcaProtocol.ControlStatus = (int) controlStatus;
                if (execMode != null)
                {
                    lcaProtocol.ExecutionMode = (ApplicationExecutionMode) execMode;
                }

                lcaProtocol.Send(theRequest);
            }
            catch (Exception e)
            {
                // log exception / throw some other one
            }
        }

        /// <summary>
        /// This method sends specific request to LCA <see cref="RequestSCSConnectionStatus"/>
        /// to get state of LCA to SCS connection.
        /// It causes LCA to send back "EventHostUnderControl" or "EventHostOffSCSControl". As a result,
        /// <see cref="OnHostUnderSCSControl(EventHostUnderControl)"/> or <see cref="OnHostOffSCSControl(EventNoControllingSCS)"/>
        /// method will be executed (with LCA invoker thread).
        ///
        /// Note: This feature is new for MF 8.1.2. Earlier LCA versions do not recognize these messages,
        /// so there will be no effect on this method call - just log record in the LCA log (if its configured).
        /// </summary> 
        /// <seealso cref="OnHostUnderSCSControl(EventHostUnderControl)"/>
        /// <seealso cref="OnHostOffSCSControl(EventNoControllingSCS)"/>
        /// <seealso cref="RequestSCSConnectionStatus"/>
        protected void DoRequestSCSConnectionStatus()
        {
            try
            {
                lcaProtocol.Send(RequestSCSConnectionStatus.Create());
            }
            catch (Exception e)
            {
                // log exception / throw some other one
            }
        }


        /// <summary>
        ///  This method will be called to notify application about successful connection to LCA.
        /// </summary> 
        /// <seealso cref="OnLCAConnectionOpened()"/>
        protected abstract void OnLCAConnectionOpened();

        /// <summary>
        /// This method will be called to notify application closed or lost connection to LCA.<br/>
        /// It may also be called in case of connection failure (exception on connection open procedure).<br/>
        /// This sample contains simple connection restore logic, which, in case of connection lost,
        /// schedules automatic reconnection with 30 seconds delay. So, this event may raise multiple times
        /// meaning that we still have no connection to the LCA.
        /// </summary> 
        /// <param name="evt">Channel closed event with reason</param>
        /// <seealso cref="OnLCAConnectionOpened()"/>
        protected abstract void OnLCAConnectionLost(EventArgs evt);


        /// <summary>
        /// This method will be called to notify application about SCS-to-LCA connection availability.
        /// </summary>
        ///<seealso cref="OnHostOffSCSControl(EventNoControllingSCS)"/>
        protected abstract void OnHostUnderSCSControl(EventHostUnderControl evt);



        /// <summary>
        /// This method will be called to notify application about absence of SCS-to-LCA connection.
        /// </summary>
        ///<seealso cref="OnHostUnderSCSControl(EventHostUnderControl)"/>
        protected abstract void OnHostOffSCSControl(EventNoControllingSCS evt);


        /// <summary>
        /// This is a callback method initiated by SCS to switch application to new execution mode.
        /// </summary>
        /// <param name="execMode">Actual application execution mode</param> 
        /// <returns>True if mode change has been finished successfully</returns>
        ///<see cref="EventChangeExecutionMode"/>    
        protected abstract bool OnChangeExecutionMode(ApplicationExecutionMode execMode);


        /// <summary>
        /// This is a callback method initiated by SCS to request application suspend.
        /// It is a first step in the "Graceful Stop" Management Framework feature.
        /// </summary>
        ///<see cref="OnChangeExecutionMode"/>
        ///<see cref="EventChangeExecutionMode"/>
        protected abstract void OnSuspendApplication();


        protected void OnLCAConnectionError(ErrorArgs evt)
        {
            // do nothing (by default)
        }

        private ITimerActionTicket lcaReopenTicket = null;

        /// <summary>
        /// Internal channel opned event handler
        /// </summary>
        private void OnChannelOpened(object sender, EventArgs args)
        {
            OnLCAConnectionOpened();
        }

        /// <summary>
        /// Internal channel closed event handler
        /// </summary>
        private void OnChannelClosed(object sender, EventArgs args)
        {
            OnLCAConnectionLost(args);
            var errorArgs = args as ErrorArgs;
            if ((errorArgs != null) && (errorArgs.Cause != null) && (ExecutionMode != ApplicationExecutionMode.Exiting) &&
                (lcaReopenTicket == null))
            {
                //try to reconnect afyer 30 seconds 
                lcaReopenTicket = TimerFactory.Scheduler.Schedule(30000, new LCAReopenTask(this));
            }
        }

        /// <summary>
        /// Internal channel error event handler
        /// </summary>
        private void OnChannelError(object sender, EventArgs args)
        {
            OnLCAConnectionError(args as ErrorArgs);
        }

        ///<summary>
        /// Internal message handler for the LCA protocol connection.
        ///</summary>
        private void OnLCAaMessage(object sender, EventArgs args)
        {
            var messageArgs = args as MessageEventArgs;
            if (messageArgs != null)
            {
                var message = messageArgs.Message;

                if (message != null)
                {
                    try
                    {
                        int referenceId = 0;
                        if (message is IReferenceable)
                        {
                            referenceId = (int) ((IReferenceable) message).RetrieveReference();
                        }

                        switch (message.Id)
                        {

                            case EventChangeExecutionMode.MessageId:
                                ApplicationExecutionMode newMode = ((EventChangeExecutionMode) message).ExecutionMode;
                                try
                                {
                                    if (OnChangeExecutionMode(newMode))
                                    {
                                        lcaProtocol.ExecutionMode = newMode;
                                    }
                                }
                                finally
                                {
                                    DoResponseExecutionModeChanged(referenceId);
                                }
                                break;

                            case EventSuspendApplication.MessageId:
                                OnSuspendApplication();
                                DoUpdateStatus(ExecutionMode, ApplicationStatus.Suspended);
                                break;

                            case EventHostUnderControl.MessageId:
                                OnHostUnderSCSControl((EventHostUnderControl) message);
                                break;

                            case EventNoControllingSCS.MessageId:
                                OnHostOffSCSControl((EventNoControllingSCS) message);
                                break;

                            case EventInitHeartbeats.MessageId:
                                // Nothing special is required here (in general) -
                                // the protocol handles the heartbeats internally.
                                break;

                            default:
                                // report/log non-handled LCA message
                                break;
                        }

                    }
                    catch (Exception e)
                    {
                        // log exception / throw some other one
                    }
                }
            }

        }

        ///<summary>
        /// Timer task, which may be scheduled in case of LCA connection lost, to start reopen procedure later.
        /// It's used internally.
        ///</summary>
        private class LCAReopenTask : ITimerAction
        {
            private AbstractServerApp application;

            public LCAReopenTask(AbstractServerApp app)
            {
                application = app;
            }

            public void OnTimer()
            {
                if (application.lcaReopenTicket != null)
                {
                    application.lcaReopenTicket.Cancel();
                    application.lcaReopenTicket = null;
                }

                if ((application.lcaProtocol != null) && !application.LCAConnected &&
                    (application.ExecutionMode != ApplicationExecutionMode.Exiting))
                {
                    try
                    {
                        application.lcaProtocol.BeginOpen();
                    }
                    catch (ProtocolException e)
                    {
                    }
                }
            }
        }
    }
}

    




